/*
* Copyright (c) 2001 Fabrice Bellard
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in
* all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
* THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
* THE SOFTWARE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/time.h>

#include <getopt.h>
#include <libavcodec/avcodec.h>
#include <libavutil/opt.h>
#include <libavutil/imgutils.h>
#include <libavutil/time.h>
#include <libavutil/frame.h>
#include <libavformat/avformat.h>
#include <libavfilter/buffersink.h>
#include <libavfilter/buffersrc.h>
#include <libswscale/swscale.h>

#include <pthread.h>

#define MAX_THREADS 32
#define SUCCESS 0
#define FAIL -1

struct list_head {
    struct list_head *next;
};

struct frame_node {
    AVFrame *frame;
    struct frame_node *next,*prev;
};

int threads = 1;
int frame_cnt;
struct codec_worker {
    const char *encoder_name;
    const char *output_file;
    const char *xcoder_params;
    const AVCodec *codec;
    AVCodecContext *codec_ctx;
    AVFilterContext *buffersink_ctx;
    AVFilterContext *buffersrc_ctx;
    AVFilterGraph *filter_graph;
    int enc_idx;
    int frame_number;
    int src_height;
    int src_width;
    int dst_height;
    int dst_width;
    int index;
    int bitrate;
    int fps;
    pthread_t tid;
    struct frame_node *head,*tail;
    pthread_mutex_t frm_mtx;
    pthread_cond_t frm_cond;
};

struct output_wh{
    int width;
    int height;
};

struct codec_worker *codec_works;

static int init_list(struct codec_worker *encode_work)
{
    encode_work->head = malloc(sizeof(struct frame_node));
    if(!encode_work->head)
    {
        printf("malloc fail\n");
        return FAIL;
    }
    encode_work->tail = malloc(sizeof(struct frame_node));
    if(!encode_work->tail)
    {
        printf("malloc fail\n");
        return FAIL;
    }
    encode_work->head->next = NULL;
    encode_work->tail->next = NULL;
    return SUCCESS;
}

static void add_frame(struct codec_worker *encode_work, const AVFrame *frame)
{
    struct frame_node *frame_adding;
    frame_adding = malloc(sizeof(struct frame_node));
    if(frame_adding == NULL)
    {
        printf("malloc fail\n");
        exit(1);
    }
    frame_adding->frame = av_frame_clone(frame);
    frame_adding->next = frame_adding->prev = NULL;
    if (!frame_adding->frame)
    {
        av_log(NULL,AV_LOG_ERROR, "Could not allocate AVFrame\n");
        exit(1);
    }

    if(encode_work->head->next == NULL)
    {
        encode_work->head->next = frame_adding;
        encode_work->tail->next = frame_adding;
    }
    else
    {
        encode_work->tail->next->next = frame_adding;
        encode_work->tail->next = frame_adding;
    }
}

static void del_frame(struct frame_node *head, struct frame_node *tail)
{
    if(head->next == NULL)
    {
        printf("list is empty!\n");
        return ;
    }
    struct frame_node *tmp_node;
    tmp_node = head->next;
    head->next = head->next->next;
    av_frame_free(&tmp_node->frame);
    tmp_node->next = NULL;
    free(tmp_node);
}

static int empty_frame(struct frame_node *head)
{
    return head->next == NULL;
}

static void setup_loglevel(const char *loglevel)
{
    int i;
    const struct { const char *name; int level; } log_levels[] =
    {
        { "quiet"  , AV_LOG_QUIET   },
        { "panic"  , AV_LOG_PANIC   },
        { "fatal"  , AV_LOG_FATAL   },
        { "error"  , AV_LOG_ERROR   },
        { "warning", AV_LOG_WARNING },
        { "info"   , AV_LOG_INFO    },
        { "verbose", AV_LOG_VERBOSE },
        { "debug"  , AV_LOG_DEBUG   },
        { "trace"  , AV_LOG_TRACE   },
    };

    for (i = 0; i < (sizeof(log_levels) / sizeof((log_levels)[0])); i++)
    {
        if (!strcmp(log_levels[i].name, loglevel))
        {
            av_log_set_level(log_levels[i].level);
        }
    }
}

/* Report processed frame count and FPS */
static void print_report(int is_last_report, int64_t start_time, int64_t cur_time)
{
    static int64_t last_time = -1;
    float t;
    char buf[1024];
    float fps;
    int i;
    int total_frame_number=0;

    if (!is_last_report)
    {
        if (last_time == -1)
        {
            last_time = cur_time;
            return;
        }
        if ((cur_time - last_time) < 10000) // do not update performance more than once every 0.5s
            return;
        last_time = cur_time;
    }

    t = (cur_time - start_time) / 1000000.0;
    for(i=0;i<threads;i++)
    {
        total_frame_number += codec_works[i].frame_number;
    }
    fps = t > 0 ? total_frame_number / t : 0;
    const char end = is_last_report ? '\n' : '\r';

    buf[0] = '\0';
    snprintf(buf + strlen(buf), sizeof(buf) - strlen(buf), "frame=%5d fps=%3.*f ",
             total_frame_number, (fps < 9.95), fps);
    av_log(NULL,AV_LOG_INFO, "%s   %c", buf, end);
}

static int init_filters(struct codec_worker *codec_work)
{
    char args[128] = { 0 };
    char name[32] = { 0 };
    int i, ret = 0;
    AVFilterInOut *inputs, *outputs, *cur;
    AVBufferSrcParameters *par;
    char filters_desc[128];
    memset(filters_desc,0,sizeof(filters_desc));
    snprintf(filters_desc,sizeof(filters_desc),"ni_quadra_hwupload=0,ni_quadra_scale=%d:%d:format=0",
             codec_work->dst_width,codec_work->dst_height);

    codec_work->filter_graph = avfilter_graph_alloc();
    if (codec_work->filter_graph == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate filter graph\n");
        goto end;
    }

    ret = avfilter_graph_parse2(codec_work->filter_graph, filters_desc, &inputs, &outputs);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to parse graph\n");
        goto end;
    }

    // link input
    cur = inputs, i = 0;
    snprintf(name, sizeof(name), "in_%d", i);
    snprintf(args, sizeof(args),
            "video_size=%dx%d:pix_fmt=%d:time_base=1/%d:pixel_aspect=0/1",
            codec_work->src_width, codec_work->src_height, AV_PIX_FMT_BGRA,
            codec_work->fps);
    ret = avfilter_graph_create_filter(&codec_work->buffersrc_ctx, avfilter_get_by_name("buffer"), name,
                                    args, NULL, codec_work->filter_graph);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot create buffer source\n");
        goto end;
    }

    ret = avfilter_link(codec_work->buffersrc_ctx, 0, cur->filter_ctx, cur->pad_idx);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to link input filter\n");
        goto end;
    }

    cur = outputs, i = 0;
    snprintf(name, sizeof(name), "out_%d", i);
    ret = avfilter_graph_create_filter(&codec_work->buffersink_ctx, avfilter_get_by_name("buffersink"),
                                        name, NULL, NULL, codec_work->filter_graph);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to create output filter: %d\n", i);
        goto end;
    }

    // connect  dst (index i) pads to one of buffer sink (index 0) pad
    ret = avfilter_link(cur->filter_ctx, cur->pad_idx, codec_work->buffersink_ctx, 0);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to link output filter: %d\n", i);
        goto end;
    }

    // configure and validate the filter graph
    ret = avfilter_graph_config(codec_work->filter_graph, NULL);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "%s failed to config graph filter\n",
               __func__);
        goto end;
    } else {
        av_log(NULL, AV_LOG_DEBUG, "%s success config graph filter %s\n",
               __func__, filters_desc);
    }

end:
    avfilter_inout_free(&inputs);
    avfilter_inout_free(&outputs);
    return ret;
}

static int init_encode(struct codec_worker *codec_work)
{
    int rc;
    // Allocate codec context for encoding
    codec_work->codec_ctx = avcodec_alloc_context3(codec_work->codec);
    if (!codec_work->codec_ctx)
    {
        av_log(NULL,AV_LOG_ERROR, "Could not allocate video codec context\n");
        return FAIL;
    }

    codec_work->codec_ctx->bit_rate = codec_work->bitrate;
    codec_work->codec_ctx->width = codec_work->dst_width; // resolution must be divisible by 2
    codec_work->codec_ctx->height = codec_work->dst_height;
    codec_work->codec_ctx->time_base = (AVRational){1, codec_work->fps};
    codec_work->codec_ctx->framerate = (AVRational){codec_work->fps, 1};
    codec_work->codec_ctx->pix_fmt = AV_PIX_FMT_NI_QUAD;
    codec_work->codec_ctx->sw_pix_fmt = AV_PIX_FMT_YUV420P;

    if ((codec_work->codec->id == AV_CODEC_ID_H264) || (codec_work->codec->id == AV_CODEC_ID_H265))
    {
        av_opt_set(codec_work->codec_ctx->priv_data, "xcoder-params", codec_work->xcoder_params, 0);
        if (codec_work->enc_idx >= 0)
            av_opt_set_int(codec_work->codec_ctx->priv_data, "enc", codec_work->enc_idx, 0);
    }
    else
    {
        av_log(NULL,AV_LOG_ERROR, "codec id %d not supported.\n", codec_work->codec->id);
        return FAIL;
    }

    // Open encoder
    rc = avcodec_open2(codec_work->codec_ctx, codec_work->codec, NULL);
    if (rc < 0)
    {
        av_log(NULL,AV_LOG_ERROR, "Could not open codec: %s\n", av_err2str(rc));
        return rc;
    }
    return SUCCESS;
}

/* Send one frame to encoder, attempt to read all available encoder packets from encoder,
writing packets to output stream. If run with with *frame=NULL, flush the encoder of
remaining packets until encoder reports EOF */
static void encode(AVCodecContext *enc_ctx, AVFrame *frame, AVPacket *pkt,
                FILE *outfile)
{
    int rc;
    int attempt_cnt = 1;
    int re_send = 0;
    static int send_cnt = 0;
    static int recv_cnt = 0;

    // Send frame to encoder. If frame=NULL, this signals end of stream
    if (frame)
    {
        frame->pict_type = AV_PICTURE_TYPE_NONE;
        av_log(NULL, AV_LOG_DEBUG, "Send %d frame %3"PRId64", type=%d\n", ++send_cnt, frame->pts, frame->pict_type);
    } else
    {
        av_log(NULL, AV_LOG_DEBUG,"Send done, flush encoder\n");
    }

resend:
    rc = avcodec_send_frame(enc_ctx, frame);
    // Encoder is too busy to receive frame, try sending frame again
    if (rc == AVERROR(EAGAIN))
    {
        av_log(NULL, AV_LOG_DEBUG, "Resending frame. Try #%d.\n", ++attempt_cnt);
        rc = 0;
        re_send = 1;
    }
    // Frame send issue occured
    else if (rc < 0)
    {
        av_log(NULL,AV_LOG_ERROR, "Error sending frame for encoding. RC=%d\n", AVUNERROR(rc));
        avcodec_free_context(&enc_ctx);
        exit(1);
    }

    /* Run this loop if a frame was sucessfully sent during avcodec_send_frame() OR
    no frame was sent because this function is in flushing mode OR
    a frame was sucessfully received by avcodec_receive_packet() */
    while (rc >= 0 || ! frame )
    {
        av_log(NULL,AV_LOG_DEBUG, "Receiving packet\n");
        rc = avcodec_receive_packet(enc_ctx, pkt);

        // No packets in encoder output buffer but not end of stream either
        if (rc == AVERROR(EAGAIN))
        {
            // If function not in flushing mode
            if (frame)
            {
                if (re_send)
                {
                    re_send = 0;
                    goto resend;
                }
                return; // Exit this function to send next packet
            }
            else
            {
                continue; // Loop in this function to flush encoder
            }
        }
        // End of stream detected
        else if (rc == AVERROR_EOF)
        {
            av_log(NULL,AV_LOG_ERROR, "Encoder EOF reached\n");
            return;
        }
        // Packet receive error occured
        else if (rc < 0)
        {
            av_log(NULL,AV_LOG_ERROR, "Error during encoding\n");
            avcodec_free_context(&enc_ctx);
            exit(1);
        }
        // Packet sucessfully received
        recv_cnt++;
        av_log(NULL,AV_LOG_DEBUG, "Write %d packet pts %3"PRId64" (size=%5d)\n",recv_cnt,
                pkt->pts, pkt->size);
        fwrite(pkt->data, 1, pkt->size, outfile);
        av_packet_unref(pkt);
    }
}

static void *encode_worker_func(void *thread_data)
{
    struct codec_worker *codec_work = (struct codec_worker *)thread_data;

    AVPacket *pkt = NULL;

    AVFrame *filt_frame = NULL;
    FILE *output_fp = NULL;
    int rc = -1;

    char outputfile[256] = {0};
    char *extensions = NULL;

    if (codec_work->codec->id == AV_CODEC_ID_H264)
    {
        extensions = "h264";
    }
    else if(codec_work->codec->id == AV_CODEC_ID_H265)
    {
        extensions = "h265";
    }
    else
    {
        av_log(NULL,AV_LOG_ERROR, "codec id %d not supported.\n", codec_work->codec->id);
        return (void *)(long)rc;
    }
    // Open outputfile
    snprintf(outputfile, sizeof(outputfile) - 1, "%s-%d-%dP.%s", codec_work->output_file,
             codec_work->index, codec_work->dst_height, extensions);
    output_fp = fopen(outputfile, "wb");
    if (!output_fp)
    {
        av_log(NULL,AV_LOG_ERROR, "Could not open outputfile: %s\n", codec_work->output_file);
        return (void *)(long)rc;
    }

    // Allocate packet for encoder output
    pkt = av_packet_alloc();
    if (!pkt)
    {
        av_log(NULL,AV_LOG_ERROR, "Could not allocate AVPacket\n");
        return (void *)(long)rc;
    }

    filt_frame = av_frame_alloc();
    if (!filt_frame)
    {
        av_log(NULL,AV_LOG_ERROR, "Could not allocate AVFrame\n");
        return (void *)(long)rc;
    }

    // Repeatedly loop input if selected by user

    while(1)
    {
        pthread_mutex_lock(&codec_work->frm_mtx);

        while(empty_frame(codec_work->head))
        {
            pthread_cond_wait(&codec_work->frm_cond, &codec_work->frm_mtx);
        }

        // codec_work->frame_number ++; // begin frame count at 0
        /* push the decoded frame into the filtergraph */
        if (av_buffersrc_add_frame_flags(codec_work->buffersrc_ctx, codec_work->head->next->frame, AV_BUFFERSRC_FLAG_KEEP_REF) < 0) {
            av_log(NULL, AV_LOG_ERROR, "Error while feeding the filtergraph\n");
            break;
        }
        av_frame_unref(codec_work->head->next->frame);
        del_frame(codec_work->head,codec_work->tail);
        pthread_mutex_unlock(&codec_work->frm_mtx);
        while(1)
        {

            rc = av_buffersink_get_frame(codec_work->buffersink_ctx, filt_frame);
            if (rc == AVERROR(EAGAIN) || rc == AVERROR_EOF || rc < 0)
            {
                break;
            }

            encode(codec_work->codec_ctx, filt_frame, pkt, output_fp);
            codec_work->frame_number ++;
            filt_frame->pict_type = AV_PICTURE_TYPE_NONE;
            av_frame_unref(filt_frame);
        }
        if(codec_work->frame_number == frame_cnt)
        {
            break;
        }
    }

    // Flush encoder
    encode(codec_work->codec_ctx, NULL, pkt, output_fp);
    fclose(output_fp);
    av_packet_free(&pkt);

    av_log(NULL,AV_LOG_INFO, "Encode done\n");
    return 0;
}

static int codec_work_init(struct codec_worker *work)
{
    if(pthread_mutex_init(&work->frm_mtx, NULL))
    {
        return FAIL;
    }
    if(pthread_cond_init(&work->frm_cond, NULL))
    {
        return FAIL;
    }

    if(init_list(work))
    {
        return FAIL;
    }

    if(init_filters(work))
    {
        return FAIL;
    }

    if(init_encode(work))
    {
        return FAIL;
    }

    return SUCCESS;
}

static void codec_work_cleanup(struct codec_worker *work)
{
    if(work)
    {
        pthread_mutex_destroy(&work->frm_mtx);
        pthread_cond_destroy(&work->frm_cond);
        avcodec_free_context(&work->codec_ctx);
        free(work->head);
        free(work->tail);

        if(work->filter_graph)
        {
            avfilter_graph_free(&(work->filter_graph));
        }
        if(work->codec_ctx)
        {
            avcodec_free_context(&(work->codec_ctx));
        }
    }
}

static void help_usage(void)
{
printf("The sample program receives bgra input, uploads swframes to 2D engine,\n"
        "and performs multi-resolution conversion coding.\n"
        "\n"
        "Usage: \n"
        "-s | --src_size              Size of input bgra in [width]x[height].\n"
        "-k | --dst_size              Size of output video in [width]x[height].\n"
        "-x | --xcoder_params         Xcoder parameters. See Integration and Programing Guide for details.\n"
        "-i | --input                 Input bgra file.\n"
        "-o | --output                Output video file.\n"
        "-c | --codec                 Codec name. [h264_ni_enc, h265_ni_enc].\n"
        "-r | --fps                   FPS of output.\n"
        "-b | --bitrate               Bitrate in bits/second.\n"
        "-e | --enc                   Encoder device index to use. See ni_rsrc_mon.\n"
        "-t | --threads               threads number.\n"
        "-g | --log_level             available debug level: warning, info, debug, trace\n"
        "-h | --help                  Help info.\n");
}

int main(int argc, char **argv)
{
    AVCodec *codec = NULL;
    AVFrame *frame = NULL;
    int bitrate = 1024000;
    int fps = 30;
    char *n;      // used for parsing width and height from --size
    const char *encoder_name = NULL;
    const char *input_file = NULL;
    const char *output_file = NULL;
    const char *xcoder_params = NULL;
    const char *loglevel = "info";
    unsigned int read_size;
    unsigned long inputfile_size;
    FILE *input_fp = NULL;
    int enc_idx = -1;
    int src_height;
    int src_width;
    int ret,i = 0,j = 0;
    int64_t start_time;
    uint8_t *src_data = NULL;  //for reading one frame data

    // Input arg handling
    int opt;
    int opt_index;

    const char *opt_string = "s:k:x:i:o:c:r:b:e:t:g:tkh";
    static struct option long_options[] =
    {
        {"src_size",           required_argument, NULL, 's'},
        {"dst_size",           required_argument, NULL, 'k'},
        {"xcoder_params",      required_argument, NULL, 'x'},
        {"inputfile",          required_argument, NULL, 'i'},
        {"outputfile",         required_argument, NULL, 'o'},
        {"codec",              required_argument, NULL, 'c'},
        {"fps",                required_argument, NULL, 'r'},
        {"bitrate",            required_argument, NULL, 'b'},
        {"enc",                required_argument, NULL, 'e'},
        {"threads",            required_argument, NULL, 't'},
        {"log_level",          required_argument, NULL, 'g'},
        {"help",               no_argument,       NULL, 'h'},
        {NULL,                 0,                 NULL,   0},
    };

struct output_wh dst_wh[MAX_THREADS];
    while ((opt = getopt_long(argc, argv, opt_string, long_options, &opt_index)) != -1)
    {
        switch (opt)
        {
            case 's':
                src_width = (int) strtol(optarg, &n, 10);
                if (*n != 'x')
                {
                    av_log(NULL,AV_LOG_ERROR, "input resolution/size arg not of format [width]x[height]\n");
                    exit(1);
                }
                src_height = atoi(n + 1);
                break;
            case 'k':
                dst_wh[i].width = (int) strtol(optarg, &n, 10);
                if (*n != 'x')
                {
                    av_log(NULL,AV_LOG_ERROR, "input resolution/size arg not of format [width]x[height]\n");
                    exit(1);
                }
                dst_wh[i].height = atoi(n + 1);
                av_log(NULL,AV_LOG_INFO,"witdth = %d,height = %d\n",dst_wh[i].width,dst_wh[i].height);
                i++;
                break;
            case 'x':
                xcoder_params = optarg;
                break;
            case 'i':
                input_file = optarg;
                break;
            case 'o':
                output_file = optarg;
                break;
            case 'c':
                encoder_name = optarg;
                break;
            case 'e':
                enc_idx = atoi(optarg);
                break;
            case 'b':
                bitrate = atoi(optarg);
                break;
            case 't':
                threads = atoi(optarg);
                break;
            case 'r':
                fps = atoi(optarg);
                break;
            case 'g':
                loglevel = optarg;
                break;
            case 'h':
                help_usage();
                exit(0);
            default:
                help_usage();
                exit(1);
        }
    }

    // Input arg checking
    if (!xcoder_params)
    {
        help_usage();
        exit(1);
    }
    if (!input_file || !output_file || !encoder_name)
    {
        av_log(NULL,AV_LOG_ERROR, "Required args (inputfile, outputfile, codec) not set.\n");
        exit(1);
    }

    av_log(NULL,AV_LOG_INFO, "inputfile:%s, outputfile:%s, codec:%s, src_width:%u, src_height:%u,enc_idx=%d.\n",\
        input_file,output_file, encoder_name,src_width, src_height,enc_idx);
    av_log(NULL,AV_LOG_INFO, "xcoder_params: %s.\n", xcoder_params);

    setup_loglevel(loglevel);

    // Register all formats, codecs, filters
#if LIBAVFORMAT_VERSION_MAJOR < 58
    av_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVCODEC_VERSION_MAJOR < 58
    avcodec_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVFILTER_VERSION_MAJOR < 7
    avfilter_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif

    codec = avcodec_find_encoder_by_name(encoder_name);
    if (!codec) {
        av_log(NULL, AV_LOG_ERROR, "Codec '%s' not found\n", encoder_name);
        exit(1);
    }

    codec_works = malloc(sizeof(struct codec_worker) * threads);
    if (codec_works == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate codec workers.\n");
        exit(1);
    }

    memset(codec_works, 0, sizeof(struct codec_worker) * threads);

    for (i = 0; i < threads; i++) {
        struct codec_worker *worker = &codec_works[i];
        worker->index = i;
        worker->encoder_name = encoder_name;
        worker->output_file = output_file;
        worker->xcoder_params = xcoder_params;
        worker->src_height = src_height;
        worker->src_width = src_width;
        worker->dst_height = dst_wh[i].height;
        worker->dst_width = dst_wh[i].width;
        worker->enc_idx = enc_idx;
        worker->bitrate = bitrate;
        worker->codec = codec;
        worker->fps = fps;
        ret = codec_work_init(worker);
        if(ret)
        {
            av_log(NULL,AV_LOG_ERROR,"init work failed!\n");
            exit(1);
        }

        ret = pthread_create(&worker->tid, NULL, encode_worker_func, worker);
        if (ret) {
            av_log(NULL, AV_LOG_ERROR, "failed to create pthread for %d: %s.\n",
                    i, strerror(errno));
            exit(1);
        }

    }
    start_time = av_gettime_relative();

    // Open input file
    input_fp = fopen(input_file, "rb");
    if (input_fp == NULL)
    {
        av_log(NULL,AV_LOG_ERROR, "Could not to open inputfile: %s\n", input_file);
        exit (1);
    }
    // Check resolution of input file
    fseek(input_fp, 0, SEEK_END);
    inputfile_size = ftell(input_fp);
    if (inputfile_size % (src_width*src_height*4) != 0)
    {
        av_log(NULL,AV_LOG_ERROR, "Size of inputfile is not integer multiple of resolution. "
            "Either input file has partial frames, or input resolution is wrong.");
        exit(1);
    }
    src_data = malloc(src_width*src_height*4);
    if(src_data == NULL)
    {
        av_log(NULL,AV_LOG_ERROR,"src_data malloc failed\n");
        exit(1);
    }
    frame_cnt = inputfile_size / (src_width*src_height*4);
    av_log(NULL,AV_LOG_INFO, "inputfile size=%lu, number of frames = %u.\n", inputfile_size, frame_cnt);

    // Repeatedly loop input if selected by user
    fseek(input_fp, 0, SEEK_SET);

    for (i = 0; i < frame_cnt; i++)
    {
        // Allocate frame object
        frame = av_frame_alloc();
        if (!frame)
        {
            av_log(NULL,AV_LOG_ERROR, "Could not allocate AVFrame\n");
            exit(1);
        }
        frame->format = AV_PIX_FMT_BGRA;
        frame->width  =src_width;
        frame->height = src_height;

        read_size = fread(src_data, 1,src_width*src_height*4,input_fp);
        if (read_size != src_width*src_height*4) {
            av_log(NULL,AV_LOG_ERROR, "Failed to read data. read_size=%u.\n", read_size);
            exit(1);
        }

        if ((av_image_fill_arrays(frame->data, frame->linesize,
                src_data, AV_PIX_FMT_BGRA,src_width, src_height, 1)) < 0) {

            av_log(NULL,AV_LOG_ERROR,"copy data to frame failed\n");
            exit(1);
        }
        frame->pts = i * frame_cnt;
        for(j = 0;j < threads; j++)
        {
            struct codec_worker *worker = &codec_works[j];
            pthread_mutex_lock(&worker->frm_mtx);
            add_frame(worker,frame);
            pthread_cond_signal(&worker->frm_cond);
            pthread_mutex_unlock(&worker->frm_mtx);
        }
        av_frame_unref(frame);
        print_report(0, start_time, av_gettime_relative());
    }

    for(j = 0;j < threads; j++)
    {
        void *res;
        struct codec_worker *worker = &codec_works[j];
        ret = pthread_join(worker->tid,&res);
        if(ret != 0)
        {
            av_log(NULL,AV_LOG_ERROR,"pthread join fail\n");
            exit(1);
        }
        codec_work_cleanup(worker);
    }
    av_frame_free(&frame);
    fclose(input_fp);
    free(src_data);
    free(codec_works);
    print_report(1, start_time, av_gettime_relative());
    return 0;
}
