/*
 * Copyright (c) 2010 Nicolas George
 * Copyright (c) 2011 Stefano Sabatini
 * Copyright (c) 2014 Andrey Utkin
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * @file
 * API example for demuxing, decoding, filtering, encoding and muxing
 * @example transcoding.c
 *
 * @added by zheng.lv@netint.ca
 * use multiple threads to run transcoding.
 */

#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>
#include <pthread.h>
#include <signal.h>
#include <assert.h>

#include <libavcodec/avcodec.h>
#include <libavformat/avformat.h>
#include <libavfilter/buffersink.h>
#include <libavfilter/buffersrc.h>
#include <libavutil/opt.h>
#include <libavutil/pixdesc.h>
#include <libavutil/time.h>
#include <libavutil/avassert.h>

#define AVFRAME_BLOCKING_COUNT 100

#define container_of(ptr, type, member) \
    ((type *)((char *)(ptr) - (size_t)&(((type *)0)->member)))

#define list_entry(ptr, type, member) \
    container_of(ptr, type, member)

#define list_first_entry(ptr, type, field) \
    list_entry((ptr)->next, type, field)

struct list_head {
    struct list_head *next, *prev;
};

typedef struct frame_node {
    struct list_head link;
    AVFrame frame;
} FrameNode;

typedef struct StreamContext {
    AVFormatContext *ifmt_ctx;
    AVCodecContext *dec_ctx;
    AVFrame *decoded_frame;
    struct list_head frame_list;
    int64_t max_duration;
    int64_t total_duration;
    int64_t last_duration;
    int64_t max_pts;
    int64_t min_pts;
    int64_t last_decoded_pts;
    int64_t last_decoded_dts;
} StreamContext;

struct common {
    pthread_mutex_t lock;
    pthread_cond_t ready_cond;
    int ready_num;
    int exit_num;
    int total_threads;
};

struct codec_worker {
    int index;
    pthread_t tid;

    pthread_cond_t consume_cond;
    pthread_cond_t product_cond;

    pthread_mutex_t frame_lock;
    unsigned int filt_frames;
    unsigned long total_decoder_frames;

    StreamContext stream_ctxes[2];
    AVFilterGraph *filter_graph;
    AVFilterContext *bufsrc_ctxes[2];
    AVFilterContext *bufsink_ctx;

    int should_exit;

    const char *decoder_name;
    const char *input_files[2];
    const char *output_dir;
    FILE *output_file;
    /* for output file name */
    int devid;
    int mode;

    struct common *common;
};

struct codec_worker *codec_workers = NULL;
int active_codec_workers = 0;
int global_stop = 0;
int print_stat = 1;

static void *codec_worker_filter_thread(void *thread_data);

static inline void INIT_LIST_HEAD(struct list_head *list)
{
    list->next = list->prev = list;
}

static inline int list_empty(const struct list_head *head)
{
    return head->next == head;
}

static inline void __list_add(struct list_head *new, struct list_head *prev, struct list_head *next)
{
    next->prev = new;
    new->next = next;
    new->prev = prev;
    prev->next = new;
}

static inline void list_add(struct list_head *_new, struct list_head *head)
{
    __list_add(_new, head, head->next);
}

static inline void list_add_tail(struct list_head *_new, struct list_head *head)
{
    __list_add(_new, head->prev, head);
}

static inline void __list_del(struct list_head *entry)
{
    entry->next->prev = entry->prev;
    entry->prev->next = entry->next;
}

static inline void list_del(struct list_head *entry)
{
    __list_del(entry);
    entry->next = entry->prev = NULL;
}

static void print_report(int is_last_report, int64_t timer_start, int64_t cur_time)
{
    int i;
    static int64_t last_time = -1;
    float t;
    char buf[1024];
    float fps;
    unsigned long frame_number = 0;

    if (!print_stat)
        return;

    if (!is_last_report) {
        if (last_time == -1) {
            last_time = cur_time;
            return;
        }
        if ((cur_time - last_time) < 500000)
            return;
        last_time = cur_time;
    }

    t = (cur_time - timer_start) / 1000000.0;

    for (i = 0; i < active_codec_workers; i++) {
        frame_number += codec_workers[i].total_decoder_frames;
    }

    fps = t > 1 ? frame_number / t : 0;
    if (print_stat) {
        const char end = is_last_report ? '\n' : '\r';

        buf[0] = '\0';
        snprintf(buf + strlen(buf), sizeof(buf) - strlen(buf), "frame=%5lu fps=%3.*f ",
                frame_number, (fps < 9.95), fps);
        if (AV_LOG_INFO > av_log_get_level()) {
            fprintf(stderr, "%s   %c", buf, end);
        } else {
            av_log(NULL, AV_LOG_INFO, "%s   %c", buf, end);
        }
        fflush(stderr);
    }
}

static int open_input_file(StreamContext* stream_ctx, const char *codec_name,
                           int devid, const char *input_file)
{
    int ret;
    AVFormatContext *ifmt_ctx = NULL;

    if ((ret = avformat_open_input(&ifmt_ctx, input_file, NULL, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot open input file\n");
        return ret;
    }

    if ((ret = avformat_find_stream_info(ifmt_ctx, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot find stream information\n");
        return ret;
    }

    //only for video
    AVStream *stream = ifmt_ctx->streams[0];
    AVCodecContext *codec_ctx;
    const AVCodec *dec;

    if ((stream->codecpar->codec_id == AV_CODEC_ID_H264 ||
            stream->codecpar->codec_id == AV_CODEC_ID_H265) &&
            codec_name) {
        dec = avcodec_find_decoder_by_name(codec_name);
        if (dec) {
            if (stream->codecpar->codec_id != dec->id) {
                av_log(NULL, AV_LOG_ERROR, "codec does not match with stream id\n");
                return AVERROR_DECODER_NOT_FOUND;
            }
        }
    } else {
        dec = avcodec_find_decoder(stream->codecpar->codec_id);
    }

    if (!dec) {
        av_log(NULL, AV_LOG_ERROR, "Failed to find decoder for stream \n");
        return AVERROR_DECODER_NOT_FOUND;
    }

    codec_ctx = avcodec_alloc_context3(dec);
    if (!codec_ctx) {
        av_log(NULL, AV_LOG_ERROR, "Failed to allocate the decoder context for stream \n");
        return AVERROR(ENOMEM);
    }
    ret = avcodec_parameters_to_context(codec_ctx, stream->codecpar);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to copy decoder parameters to input "
               "decoder context for stream\n");
        return ret;
    }

    av_log(NULL, AV_LOG_DEBUG, "# stream.time_base=%d/%d avg_frame_rate=%d/%d\n",
            stream->time_base.num, stream->time_base.den,
            stream->avg_frame_rate.num, stream->avg_frame_rate.den);

    if (!strncmp(dec->name, "h264_ni", strlen("h264_ni")) ||
        !strncmp(dec->name, "h265_ni", strlen("h265_ni"))) {
        char str_devid[4] = {0};
        snprintf(str_devid, sizeof(str_devid), "%d", devid);
        av_opt_set(codec_ctx->priv_data, "dec", str_devid, 0);
    }

    if (codec_ctx->codec_type == AVMEDIA_TYPE_VIDEO)
        codec_ctx->framerate = av_guess_frame_rate(ifmt_ctx, stream, NULL);

    ret = avcodec_parameters_from_context(stream->codecpar, codec_ctx);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to copy decoder parameters from codec context for stream");
        return ret;
    }

    /* Reencode video & audio and remux subtitles etc. */
    if (codec_ctx->codec_type == AVMEDIA_TYPE_VIDEO) {
        /* Open decoder */
        ret = avcodec_open2(codec_ctx, dec, NULL);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "Failed to open decoder for stream\n");
            return ret;
        }
    }

    av_dump_format(ifmt_ctx, 0, input_file, 0);

    stream_ctx->dec_ctx = codec_ctx;
    stream_ctx->last_decoded_pts = AV_NOPTS_VALUE;
    stream_ctx->last_decoded_pts = AV_NOPTS_VALUE;
    stream_ctx->ifmt_ctx = ifmt_ctx;

    return 0;
}

static void send_filter_frame(struct codec_worker *worker, AVFrame *frame, int index)
{
    FrameNode *fn;
    StreamContext *stream_ctx = &worker->stream_ctxes[index];

    fn = (FrameNode *) malloc(sizeof(*fn));
    av_assert0(fn);
    av_frame_move_ref(&fn->frame, frame);

    pthread_mutex_lock(&worker->frame_lock);
    list_add_tail(&fn->link, &stream_ctx->frame_list);
    pthread_cond_signal(&worker->consume_cond);
    worker->filt_frames++;
    if (worker->filt_frames >= AVFRAME_BLOCKING_COUNT) {
        do {
            pthread_cond_wait(&worker->product_cond, &worker->frame_lock);
            if ((worker->filt_frames < AVFRAME_BLOCKING_COUNT) ||
                    worker->should_exit) {
                break;
            }
        } while (1);
    }
    worker->total_decoder_frames++;
    pthread_mutex_unlock(&worker->frame_lock);
}

static int __decode(struct codec_worker *worker, AVPacket *packet, int index)
{
    int ret = 0;
    AVFrame *frame;
    AVPacket avpkt;
    StreamContext *stream_ctx = &worker->stream_ctxes[index];

    if (packet) {
        avpkt = *packet;
    }

    ret = avcodec_send_packet(stream_ctx->dec_ctx, packet ? &avpkt : NULL);
    if (ret < 0 && ret != AVERROR_EOF) {
        av_log(NULL, AV_LOG_ERROR, "# failed to send packet.\n");
        goto end;
    }

    frame = stream_ctx->decoded_frame;
    while (ret >= 0 || !packet) {
        ret = avcodec_receive_frame(stream_ctx->dec_ctx, frame);
        if (ret < 0) {
            if (ret != AVERROR(EAGAIN) && ret != AVERROR_EOF) {
                av_log(NULL, AV_LOG_ERROR, "failed to receive frame\n");
            }
            goto end;
        }

        if (frame->best_effort_timestamp != AV_NOPTS_VALUE) {
            frame->best_effort_timestamp += stream_ctx->last_duration;
        }
        frame->pts = frame->best_effort_timestamp;
#if LIBAVCODEC_VERSION_MAJOR >= 61 //7.0
        if (stream_ctx->max_duration < frame->duration) {
            stream_ctx->max_duration = frame->duration;
        }
#else
        if (stream_ctx->max_duration < frame->pkt_duration) {
            stream_ctx->max_duration = frame->pkt_duration;
        }
#endif
        if (frame->pts != AV_NOPTS_VALUE) {
            if (stream_ctx->max_pts < frame->pts) {
                stream_ctx->max_pts = frame->pts;
            }
            if (stream_ctx->min_pts > frame->pts) {
                stream_ctx->min_pts = frame->pts;
            }
        }

        if (packet) {
            av_log(NULL, AV_LOG_DEBUG, "decoded frame: stream=%d, pts=%ld, "
                   "dts=%ld, duration=%ld, best_effort=%ld\n",
                    index, frame->pts, frame->pkt_dts,
#if LIBAVCODEC_VERSION_MAJOR >= 61 //7.0
                    frame->duration,
#else
                    frame->pkt_duration,
#endif
                    frame->best_effort_timestamp);
        } else {
            av_log(NULL, AV_LOG_DEBUG, "flush frame: stream=%d, pts=%ld, "
                   "dts=%ld, duration=%ld, best_effort=%ld\n",
                    index, frame->pts, frame->pkt_dts,
#if LIBAVCODEC_VERSION_MAJOR >= 61 //7.0
                    frame->duration,
#else
                    frame->pkt_duration,
#endif
                   frame->best_effort_timestamp);
        }

        if ((stream_ctx->last_decoded_pts != AV_NOPTS_VALUE) &&
                (frame->pts == stream_ctx->last_decoded_pts)) {
            av_log(NULL, AV_LOG_ERROR, "flush decoder: same pts!!!\n");
        }
        if ((stream_ctx->last_decoded_dts != AV_NOPTS_VALUE) &&
                (frame->pkt_dts == stream_ctx->last_decoded_dts)) {
            av_log(NULL, AV_LOG_ERROR, "flush decoder: same dts!!!\n");
        }

#if LIBAVCODEC_VERSION_MAJOR >= 61 //7.0
        stream_ctx->total_duration += frame->duration;
#else
        stream_ctx->total_duration += frame->pkt_duration;
#endif
        stream_ctx->last_decoded_pts = frame->pts;
        stream_ctx->last_decoded_dts = frame->pkt_dts;

        send_filter_frame(worker, frame, index);
    }

end:
    return ret;
}

static int filter_frame(struct codec_worker *worker)
{
    int i, ret;
    FrameNode *fn;
    AVFrame *frame;
    int64_t pts = AV_NOPTS_VALUE;

    for (i = 0; i < 2; i++) {
        StreamContext *stream_ctx = &worker->stream_ctxes[i];

        pthread_mutex_lock(&worker->frame_lock);

        fn = list_first_entry(&stream_ctx->frame_list, FrameNode, link);
        av_assert0(fn != NULL);
        list_del(&fn->link);
        worker->filt_frames--;
        if (worker->filt_frames < AVFRAME_BLOCKING_COUNT) {
            pthread_cond_signal(&worker->product_cond);
        }

        pthread_mutex_unlock(&worker->frame_lock);

        frame = &fn->frame;
        // Force PTS consistence
        if (i == 0) {
            pts = frame->pts;
        } else {
            frame->pts = pts;
        }

        av_log(NULL, AV_LOG_DEBUG, "frame pixel=%dx%d, crop=%ld/%ld/%ld/%ld linsize %d/%d/%d\n",
               frame->width, frame->height, frame->crop_top,
               frame->crop_bottom, frame->crop_left, frame->crop_right,
               frame->linesize[0], frame->linesize[1], frame->linesize[2]);

        av_log(NULL, AV_LOG_DEBUG, "ref count %d/%d/%d\n",
               frame->buf[0] ? av_buffer_get_ref_count(frame->buf[0]) : -1,
               frame->buf[1] ? av_buffer_get_ref_count(frame->buf[1]) : -1,
               frame->buf[2] ? av_buffer_get_ref_count(frame->buf[2]) : -1);

        frame->pict_type = AV_PICTURE_TYPE_NONE;
        av_log(NULL, AV_LOG_DEBUG, "decode frame %3"PRId64", type=%d\n", frame->pts, frame->pict_type);

        ret = av_buffersrc_add_frame_flags(worker->bufsrc_ctxes[i], frame, AV_BUFFERSRC_FLAG_KEEP_REF);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "failed to add frame to buffersrc\n");
            return ret;
        }

        av_frame_unref(frame);
        free(fn);
    }

    for (; ;) {
        AVFrame filt_frame;

        ret = av_buffersink_get_frame(worker->bufsink_ctx, &filt_frame);
        if (ret == AVERROR(EAGAIN) || ret == AVERROR_EOF) {
            return 0;
        } else if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "failed to get frame from buffersink\n");
            break;
        }

        av_log(NULL, AV_LOG_DEBUG, "filt frame pixel=%dx%d, crop=%ld/%ld/%ld/%ld, linesize=%d/%d/%d\n",
               filt_frame.width, filt_frame.height, filt_frame.crop_top,
               filt_frame.crop_bottom, filt_frame.crop_left, filt_frame.crop_right,
               filt_frame.linesize[0], filt_frame.linesize[1], filt_frame.linesize[2]);

        av_log(NULL, AV_LOG_DEBUG, "filt ref count %d/%d/%d\n",
               filt_frame.buf[0] ? av_buffer_get_ref_count(filt_frame.buf[0]) : -1,
               filt_frame.buf[1] ? av_buffer_get_ref_count(filt_frame.buf[1]) : -1,
               filt_frame.buf[2] ? av_buffer_get_ref_count(filt_frame.buf[2]) : -1);

        if (worker->output_file != NULL) {
            for (i = 0; i < 3; i++) {
                fwrite(filt_frame.data[i], filt_frame.linesize[i],
                       filt_frame.height, worker->output_file);
            }
        }
    }

    return ret;
}

static int decoder_thread_run(struct codec_worker *worker)
{
    AVPacket packet = { 0 };
    int i, ret, flag;

    while (!worker->should_exit) {
        flag = AVERROR_EOF;
        for (i = 0; i < 2; i++) {
            StreamContext *stream_ctx = &worker->stream_ctxes[i];
            ret = av_read_frame(stream_ctx->ifmt_ctx, &packet);
            if (packet.stream_index == 0) {
                ret = __decode(worker, ret == AVERROR_EOF ? NULL : &packet, i);
                if (ret == AVERROR_EOF) {
                    ret = 0;
                } else {
                    flag = ret;
                    if (ret != AVERROR(EAGAIN)) {
                        goto end;
                    }
                }
            }
            av_packet_unref(&packet);
        }

        if (flag == AVERROR_EOF) {
            break;
        }
    }

    for (i = 0; i < 2; i++) {
        StreamContext *stream_ctx = &worker->stream_ctxes[i];
        av_log(NULL, AV_LOG_INFO, "flush decoder: stream=%d,"
               "total_duration=%ld,max_duration=%ld.\n",
               i, stream_ctx->total_duration, stream_ctx->max_duration);
        stream_ctx->last_duration = stream_ctx->total_duration;
        stream_ctx->last_duration += stream_ctx->max_duration;
    }

    av_log(NULL, AV_LOG_INFO, "end of decoder thread.\n");
end:
    return ret;
}

static int init_codec_worker(struct codec_worker *worker)
{
    int i, ret;

    for (i = 0; i < 2; i++) {
        StreamContext *stream_ctx = &worker->stream_ctxes[i];
        INIT_LIST_HEAD(&stream_ctx->frame_list);
    }

    ret = pthread_mutex_init(&worker->frame_lock, NULL);
    if (ret) {
        goto fail_init_frame_lock;
    }

    ret = pthread_cond_init(&worker->consume_cond, NULL);
    if (ret) {
        goto fail_init_consume_cond;
    }

    ret = pthread_cond_init(&worker->product_cond, NULL);
    if (ret) {
        goto fail_init_product_cond;
    }

    return 0;

fail_init_product_cond:
    pthread_cond_destroy(&worker->consume_cond);
fail_init_consume_cond:
    pthread_mutex_destroy(&worker->frame_lock);
fail_init_frame_lock:

    return ret;
}

static void cleanup_codec_worker(struct codec_worker *worker)
{
    pthread_cond_destroy(&worker->product_cond);
    pthread_cond_destroy(&worker->consume_cond);
    pthread_mutex_destroy(&worker->frame_lock);
}

static int filter_graph_open(struct codec_worker *worker)
{
    int i, ret;
    int width, height;
    AVRational sar;
    AVFilterInOut *inputs, *outputs, *cur;
    StreamContext *stream_ctx;
    char filt_desc[256] = {0};

    stream_ctx = &worker->stream_ctxes[0];
    width = stream_ctx->dec_ctx->width;
    height = stream_ctx->dec_ctx->height;
    sar = stream_ctx->dec_ctx->sample_aspect_ratio;

    worker->filter_graph = avfilter_graph_alloc();
    if (worker->filter_graph == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate filter graph\n");
        goto out;
    }

    snprintf(filt_desc, sizeof(filt_desc), "ni_quadra_yuv420to444=mode=%d", worker->mode);
    ret = avfilter_graph_parse2(worker->filter_graph, filt_desc, &inputs, &outputs);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to parse graph\n");
        goto out;
    }

    for (cur = inputs, i = 0; cur; cur = cur->next, i++) {
        char args[512] = { 0 };
        char name[256] = { 0 };

        snprintf(name, sizeof(name), "in_%d", i);
        snprintf(args, sizeof(args), "video_size=%dx%d:pix_fmt=%d:time_base=%d/%d:pixel_aspect=%d/%d",
                 width, height, AV_PIX_FMT_YUV420P, 1, 30, sar.num, sar.den);
        av_log(NULL, AV_LOG_INFO, "input filter args: %s\n", args);
        ret = avfilter_graph_create_filter(&worker->bufsrc_ctxes[i],
                                           avfilter_get_by_name("buffer"), name,
                                           args, NULL, worker->filter_graph);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "failed to create input filter: %d\n", i);
            goto out;
        }

        ret = avfilter_link(worker->bufsrc_ctxes[i], 0, cur->filter_ctx, cur->pad_idx);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "failed to link input filter: %d\n", i);
            goto out;
        }
    }

    ret = avfilter_graph_create_filter(&worker->bufsink_ctx,
                                       avfilter_get_by_name("buffersink"),
                                       "out_0", NULL, NULL, worker->filter_graph);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to create output filter: %d", i);
        goto out;
    }

    ret = avfilter_link(outputs->filter_ctx, outputs->pad_idx, worker->bufsink_ctx, 0);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to link output filter: %d\n", i);
        goto out;
    }

    ret = avfilter_graph_config(worker->filter_graph, NULL);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to config graph filter\n");
        goto out;
    }

out:
    avfilter_inout_free(&inputs);
    avfilter_inout_free(&outputs);
    return ret;
}

static void *codec_worker_decoder_thread(void *thread_data)
{
    struct codec_worker *worker = (struct codec_worker *)thread_data;
    StreamContext *stream_ctx;
    int i, ret;
    pthread_t tid;
    void *result;

    pthread_mutex_lock(&worker->common->lock);
    worker->common->ready_num++;
    if (worker->common->ready_num >= worker->common->total_threads) {
        pthread_cond_signal(&worker->common->ready_cond);
    }
    pthread_mutex_unlock(&worker->common->lock);

    for (i = 0; i < 2; i++) {
        stream_ctx = &worker->stream_ctxes[i];
        ret = open_input_file(stream_ctx, worker->decoder_name, worker->devid,
                              worker->input_files[i]);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "fail to open input file %s.\n",
                   worker->input_files[i]);
            goto end;
        }

        stream_ctx->decoded_frame = av_frame_alloc();
        if (!stream_ctx->decoded_frame) {
            av_log(NULL, AV_LOG_ERROR, "failed to allocate decoded frame for codec worker %d\n", i);
            goto end;
        }
    }

    ret = pthread_create(&tid, NULL, codec_worker_filter_thread, worker);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "failed to create pthread for %d: %s.\n",
               worker->index, strerror(errno));
        goto end;
    }

    ret = decoder_thread_run(worker);
    if (ret < 0) {
        goto end;
    }

    av_log(NULL, AV_LOG_INFO, "decoder normal shutdown.\n");

end:
    // Tell filter thread to exit.
    pthread_mutex_lock(&worker->frame_lock);
    if (!worker->should_exit) {
        worker->should_exit = 1;
    }
    pthread_cond_signal(&worker->consume_cond);
    pthread_mutex_unlock(&worker->frame_lock);

    // Wait for filter thread termination.
    pthread_join(tid, &result);
    if ((long)result != 0) {
        av_log(NULL, AV_LOG_ERROR, "filter pthread %d result=0x%lx.\n",
            worker->index, (long)result);
    }

    // Release resouces.
    for (i = 0; i < 2; i++) {
        stream_ctx = &worker->stream_ctxes[i];
        if (stream_ctx->ifmt_ctx) {
            av_frame_free(&stream_ctx->decoded_frame);
            avcodec_free_context(&stream_ctx->dec_ctx);
            avformat_close_input(&stream_ctx->ifmt_ctx);
        }
    }

    av_log(NULL, AV_LOG_INFO, "decoder exit: ret=0x%x.\n", -ret);
    pthread_mutex_lock(&worker->common->lock);
    worker->common->exit_num++;
    pthread_mutex_unlock(&worker->common->lock);

    return (void *)((long)ret);
}

static inline int codec_frame_list_empty(struct codec_worker *worker)
{
    int i;
    for (i = 0; i < 2; i++) {
        StreamContext *stream_ctx = &worker->stream_ctxes[i];
        if (list_empty(&stream_ctx->frame_list)) {
            return 1;
        }
    }
    return 0;
}

static void *codec_worker_filter_thread(void *thread_data)
{
    struct codec_worker *worker = (struct codec_worker *)thread_data;
    int ret;

    ret = filter_graph_open(worker);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to open filter graph\n");
        goto end;
    }

    if (worker->output_dir) {
        char output[256] = { 0 };
        snprintf(output, sizeof(output) - 1, "%s/output-%d.yuv", worker->output_dir,
                 worker->common->total_threads - worker->index - 1);
        worker->output_file = fopen(output, "wb");
        if (!worker->output_file) {
            av_log(NULL, AV_LOG_ERROR, "fail to open output file.\n");
            goto end;
        }
    }

    while (1) {
        pthread_mutex_lock(&worker->frame_lock);

        while (!worker->should_exit && codec_frame_list_empty(worker)) {
            // Wait for available decoded frames
            pthread_cond_wait(&worker->consume_cond, &worker->frame_lock);
        }

        if (codec_frame_list_empty(worker) && worker->should_exit) {
            // End of all frame filtering
            pthread_mutex_unlock(&worker->frame_lock);
            break;
        }

        pthread_mutex_unlock(&worker->frame_lock);

        // YUV420p to YUV444p
        ret = filter_frame(worker);
        if (ret < 0) {
            goto end;
        }
    }

end:
    av_log(NULL, AV_LOG_INFO, "filter exit: ret=0x%x.\n", -ret);

    if (worker->output_file){
       fclose(worker->output_file);
    }
    pthread_mutex_lock(&worker->common->lock);
    worker->common->exit_num++;
    pthread_mutex_unlock(&worker->common->lock);

    return (void *)((long)ret);
}

static void sigint_handler(int signo)
{
    global_stop = 1;
    av_log(NULL, AV_LOG_INFO, "%s().\n", __func__);
}

static struct common *alloc_common(void)
{
    struct common *common;
    int ret;

    common = malloc(sizeof(struct common));
    if (common == NULL) {
        return NULL;
    }

    memset(common, 0, sizeof(struct common));

    ret = pthread_mutex_init(&common->lock, NULL);
    if (ret) {
        goto fail_init_lock;
    }

    ret = pthread_cond_init(&common->ready_cond, NULL);
    if (ret) {
        goto fail_init_ready_cond;
    }

    return common;

fail_init_ready_cond:
    pthread_mutex_destroy(&common->lock);
fail_init_lock:
    free(common);
    return NULL;
}

static void help_usage(void)
{
    printf("This example would start multiple threads specified at once.\n"
           "When the thread is specified to 1, it actually opens 3 thread, 2 decoder threads and\n"
           "one filter thread 'yuv420to444'\n"
           "The output file is named 'output-%%{thread index}.yuv' and stored\n"
           "in output directory specified by parameter.\n"
           "\n"
           "The filter thread calculates and prints fps in the front until\n"
           "there is an ctrl-c interrupt or all the threads are done.\n"
           "\n"
           "Usage example:\n"
           "./ni_multi_thread_yuv420to444 -d h264_ni_quadra_dec -f ~/input0.h264 -s ~/input1.h264 \\n"
           "-o ./output/\n"
           "\n"
           "Note:\n"
           "input0 and input1 must encoded with the same encoding parameters after the 'yuv444to420' filter;\n"
           "the frame numbers of input0/input1 must be equal;\n"
           "if an error or eof occurs on any of the decoding threads, the filter will exits.\n"
           "\n"
           "Params:\n"
               "-f | --input0            the first input file path.\n"
               "-s | --input1            the second input file path.\n"
               "-m | --mode              mode 0 have better PSNR and 1 can decode as 420.\n"
               "-o | --output_dir        output directory path. if not specified, the output will be empty.\n"
               "-d | --decoder           decoder name.\n"
               "-t | --threads           number of codec threads.\n"
               "-n | --devid             device id.\n"
               "-v | --loglevel          available debug level: warning, info, debug, trace.\n"
               "-h | --help              print this help information.\n");
}

static void setup_loglevel(char *loglevel)
{
    if (loglevel) {
        if (!strcmp(loglevel, "warning")) {
            av_log_set_level(AV_LOG_WARNING);
        } else if (!strcmp(loglevel, "info")) {
            av_log_set_level(AV_LOG_INFO);
        } else if (!strcmp(loglevel, "debug")) {
            av_log_set_level(AV_LOG_DEBUG);
        } else if (!strcmp(loglevel, "trace")) {
            av_log_set_level(AV_LOG_TRACE);
        } else {
            av_log_set_level(AV_LOG_INFO);
        }
    } else {
        av_log_set_level(AV_LOG_INFO);
    }
}

int main(int argc, char **argv)
{
    int ret;
    char *input_file_0 = NULL;
    char *input_file_1 = NULL;
    char *output_dir = NULL;
    char *decoder_name = NULL;
    struct common *common = NULL;
    void *result;
    int mode = 0;
    int threads = 1;
    int state = EXIT_SUCCESS;
    int devid = 0;
    int i;
    int64_t timer_start;
    int opt;
    int opt_index;
    char *loglevel = NULL;
    const char *opt_string = "f:s:m:o:d:t:n:l:v:a:h";
    static struct option long_options[] = {
        {"input0",         required_argument, NULL, 'f'},
        {"input1",         required_argument, NULL, 's'},
        {"mode",           required_argument, NULL, 'm'},
        {"output_dir",     required_argument, NULL, 'o'},
        {"decoder",        required_argument, NULL, 'd'},
        {"threads",        required_argument, NULL, 't'},
        {"devid",          required_argument, NULL, 'n'},
        {"loglevel",       required_argument, NULL, 'v'},
        {"help",           no_argument,       NULL, 'h'},
        { NULL,            0,                 NULL,  0 },
    };

    while ((opt = getopt_long(argc, argv, opt_string, long_options, &opt_index)) != -1) {
        switch (opt) {
            case 'd':
                decoder_name = optarg;
                break;
            case 'f':
                input_file_0 = optarg;
                break;
            case 'm':
                mode = atoi(optarg);
                break;
            case 's':
                input_file_1= optarg;
                break;
            case 'o':
                output_dir = optarg;
                break;
            case 't':
                threads = atoi(optarg);
                break;
            case 'n':
                devid = atoi(optarg);
                break;
            case 'v':
                loglevel = optarg;
                break;
                help_usage();
                return EXIT_SUCCESS;
            default:
                help_usage();
                return EXIT_FAILURE;
        }
    }

    setup_loglevel(loglevel);

    if (input_file_0 == NULL || input_file_1 == NULL) {
        av_log(NULL, AV_LOG_ERROR, "input file name must be specified.\n");
        return EXIT_FAILURE;
    }

    if (output_dir == NULL) {
        output_dir = ".";
    }

    if (threads <= 0 && threads > 128) {
        av_log(NULL, AV_LOG_ERROR, "invalid number of thread worker.\n");
        return EXIT_FAILURE;
    }

    if (decoder_name) {
        av_log(NULL, AV_LOG_INFO, "decoder_name: %s.\n", decoder_name);
    }

    codec_workers = malloc(sizeof(struct codec_worker) * threads);
    if (codec_workers == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate codec workers.\n");
        state = EXIT_FAILURE;
        goto end;
    }
    memset(codec_workers, 0, sizeof(struct codec_worker) * threads);

    common = alloc_common();
    if (common == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate common data.\n");
        state = EXIT_FAILURE;
        goto end;
    }

    // Register all formats, codecs, filters
#if LIBAVFORMAT_VERSION_MAJOR < 58
    av_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVCODEC_VERSION_MAJOR < 58
    avcodec_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVFILTER_VERSION_MAJOR < 7
    avfilter_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif

    common->total_threads = threads;

    for (i = 0; i < threads; i++) {
        struct codec_worker *worker = &codec_workers[i];
        worker->index = i;
        worker->mode = mode;
        worker->input_files[0] = input_file_0;
        worker->input_files[1] = input_file_1;
        worker->output_dir = output_dir;
        worker->decoder_name = decoder_name;
        worker->common = common;
        worker->devid  = devid;

        ret = init_codec_worker(worker);
        if (ret) {
            av_log(NULL, AV_LOG_ERROR, "failed init codec worker %d.\n", i);
            state = EXIT_FAILURE;
            goto end;
        }

        ret = pthread_create(&worker->tid, NULL, &codec_worker_decoder_thread, worker);
        if (ret) {
            av_log(NULL, AV_LOG_ERROR, "failed to create codec thread %d: %s.\n", i, strerror(ret));
            state = EXIT_FAILURE;
            goto end;
        }

        active_codec_workers++;
    }

    pthread_mutex_lock(&common->lock);
    while (common->ready_num < common->total_threads)
        pthread_cond_wait(&common->ready_cond, &common->lock);
    pthread_mutex_unlock(&common->lock);

    if (SIG_ERR == signal(SIGINT, sigint_handler)) {
        av_log(NULL, AV_LOG_ERROR, "failed to set signal.\n");
        state = EXIT_FAILURE;
        goto end;
    }

    timer_start = av_gettime_relative();
    while (global_stop == 0) {
        if (common->exit_num == active_codec_workers * 2) {
            break;
        }
        sleep(1);

        print_report(0, timer_start, av_gettime_relative());
    }
    print_report(1, timer_start, av_gettime_relative());

    av_log(NULL, AV_LOG_INFO, "main thread is going to exit.\n");

end:
    for (i = 0; i < active_codec_workers; i++) {
        struct codec_worker *worker = &codec_workers[i];

        av_log(NULL, AV_LOG_INFO, "pthread %d ready to exit.\n", worker->index);

        pthread_mutex_lock(&worker->frame_lock);
        worker->should_exit = 1;
        pthread_cond_signal(&worker->product_cond);
        pthread_mutex_unlock(&worker->frame_lock);
        if (pthread_join(worker->tid, &result) == 0) {
            if ((long)result != 0) {
                state = EXIT_FAILURE;
            }
        }

        cleanup_codec_worker(worker);
    }

    if (state < 0)
        av_log(NULL, AV_LOG_ERROR, "Error occurred: %s\n", av_err2str(ret));

    pthread_mutex_destroy(&common->lock);
    pthread_cond_destroy(&common->ready_cond);
    free(common);
    free(codec_workers);

    av_log(NULL, AV_LOG_INFO, "EXIT.. state=0x%x.\n", state);
    return state ? 1 : 0;
}
