/*******************************************************************************
 *
 * Copyright (C) 2022 NETINT Technologies
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 ******************************************************************************/

/*!*****************************************************************************
 *  \file   ni_jpeg_test.c
 *
 *  \brief  Application for performing jpeg processing with libxcoder API.
 *          Its code provides examples on how to programatically use libxcoder
 *          API.
 ******************************************************************************/

#ifdef _WIN32
#include <io.h>
#include "ni_getopt.h"
#elif __linux__ || __APPLE__
#define _POSIX_C_SOURCE 200809L
#include <getopt.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/poll.h>
#endif

#include <fcntl.h>
#include <errno.h>
#include "ni_test_utils.h"
#include "ni_p2p_ioctl.h"

#if defined(LRETURN)
#undef LRETURN
#define LRETURN goto end;
#undef END
#define END                                     \
  end:
#else
#define LRETURN goto end;
#define END                                     \
  end:
#endif

#define APP0 0xE0
#define SOF0 0xC0

typedef struct _ni_filter_params_t
{
    bool p2p;
    int scale_width;
    int scale_height;
    int pad_width;
    int pad_height;
    int format;
} ni_filter_params_t;

typedef struct _ni_disp_buffer
{
    int fd;
    int bar;
    uint8_t *data;
    unsigned long len;
    volatile uint8_t *mmap_data;
} disp_buffer_t;

typedef enum
{
    DECODE_START = 0,
    DECODE_WAIT = 1,
    DECODE_READ = 2,
    DECODE_END = 3
}decod_status_t;

volatile int send_fin_flag = 0, receive_fin_flag = 0;
volatile unsigned int number_of_frames = 0;
struct timeval start_time, previous_time, current_time;
static uint8_t *g_file_cache = NULL;
static uint8_t *g_curr_cache_pos = NULL;
volatile uint64_t total_file_size = 0;
const char *filename, *outputfile;
int pfs;

int write_dmabuf_data(uint8_t *pdata, FILE *fp, int width, int height,
                      int format)
{
    if (!pdata || !fp)
        return -1;

    switch (format)
    {
        case GC620_I420:
        {
            int write_width = width;
            int write_height = height;
            int plane_width;
            int plane_height;
            int j;
            uint8_t *src = pdata;

            // write Y stride
            if (fwrite(pdata, write_width * write_height, 1, fp) != 1)
            {
                fprintf(stderr, "Error: writing Y stride error!\n");
                fprintf(stderr, "Error: ferror rc = %d\n", ferror(fp));
                return -1;
            }

            int width_aligned = (((width + 127) / 128) * 128);
            int height_aligned = ((height + 1) / 2) * 2;
            int luma_size = width_aligned * height_aligned;
            pdata = src + luma_size;

            // write U stride
            plane_height = height / 2;
            plane_width = (((int)(write_width) / 2 + 127) / 128) * 128;
            write_height /= 2;
            write_width /= 2;
            for (j = 0; j < plane_height; j++)
            {
                if (j < write_height && fwrite(pdata, write_width, 1, fp) != 1)
                {
                    fprintf(stderr,
                            "Error: writing U stride: height %d error!\n",
                            height);
                    fprintf(stderr, "Error: ferror rc = %d\n", ferror(fp));
                    return -1;
                }
                pdata += plane_width;
            }

            int chroma_b_size;
            int chroma_width_aligned = ((((width / 2) + 127) / 128) * 128);
            int chroma_height_aligned = height_aligned / 2;
            chroma_b_size = chroma_width_aligned * chroma_height_aligned;

            // write V stride
            pdata = src + luma_size + chroma_b_size;

            for (j = 0; j < plane_height; j++)
            {
                if (j < write_height && fwrite(pdata, write_width, 1, fp) != 1)
                {
                    fprintf(stderr,
                            "Error: writing V stride: height %d error!\n",
                            height);
                    fprintf(stderr, "Error: ferror rc = %d\n", ferror(fp));
                    return -1;
                }
                pdata += plane_width;
            }
            break;
        }
        case GC620_RGBA8888:
        {
            if (fwrite(pdata, width * height * 4, 1, fp) != 1)
            {
                fprintf(stderr, "Error: writing rgba data error!\n");
                fprintf(stderr, "Error: ferror rc = %d\n", ferror(fp));
                return -1;
            }
            break;
        }
        case GC620_RGB888_PLANAR:
        {
            int stride_size = NI_VPU_ALIGN32(width * height);
            int i;
            for (i = 0; i < 3; i++)
            {
                if (fwrite(pdata, width * height, 1, fp) != 1)
                {
                    fprintf(stderr, "Error: writing bgrp data error!\n");
                    fprintf(stderr, "Error: ferror rc = %d\n", ferror(fp));
                    return -1;
                }
                pdata += stride_size;
            }
            break;
        }

        default:
            break;
    }

    if (fflush(fp))
    {
        fprintf(stderr, "Error: writing data frame flush failed! errno %d\n",
                errno);
    }

    return 0;
}

int calc_frame_buffer_size(int width, int height, int format)
{
    int width_aligned = width;
    int height_aligned = height;
    int data_len = 0;
    switch (format)
    {
        case GC620_I420:
        {
            width_aligned = ((width + 127) / 128) * 128;
            height_aligned = ((height + 1) / 2) * 2;
            int luma_size = width_aligned * height_aligned;
            int chroma_b_size;
            int chroma_r_size;
            int chroma_width_aligned = (((width / 2) + 127) / 128) * 128;
            int chroma_height_aligned = height_aligned / 2;
            chroma_b_size = chroma_r_size =
                chroma_width_aligned * chroma_height_aligned;
            data_len = luma_size + chroma_b_size + chroma_r_size;
            break;
        }
        case GC620_RGBA8888:
        {
            data_len = width * height * 4;
            break;
        }
        case GC620_RGB888_PLANAR:
        {
            data_len = NI_VPU_ALIGN32(width * height) * 3;
            break;
        }

        default:
            break;
    }
    return data_len;
}

#ifndef _WIN32
/*!*****************************************************************************
 *  \brief  Read dmabuf data(4k aligned) to cpu for test.
 *
 *  \param
 *
 *  \return 0 if successful, < 0 otherwise
 ******************************************************************************/
int read_from_dmabuf(ni_session_context_t *p_ctx, niFrameSurface1_t *p_surface,
                     disp_buffer_t *disp, int format, FILE *fp)
{
    if (!disp)
    {
        printf("Error: Invalid input params\n");
        return -1;
    }
    int ret = 0;
    int data_len = calc_frame_buffer_size(p_surface->ui16width,
                                          p_surface->ui16height, format);
    data_len = (data_len + 4095) & ~4095;   // must be 4096 bytes aligned
    if (!data_len)
    {
        printf("Error: read size is 0!\n");
        return -1;
    }

    disp->fd = -1;
    disp->mmap_data = MAP_FAILED;
    disp->len = data_len;
    disp->bar = 4;

    ret = ni_scaler_p2p_frame_acquire(p_ctx, p_surface, data_len);
    if (ret != 0)
    {
        printf("failed to export dma buf\n");
        return -1;
    }
    disp->fd = p_surface->dma_buf_fd;

    //mmap
    disp->mmap_data =
        mmap(0, data_len, PROT_READ | PROT_WRITE, MAP_SHARED, disp->fd, 0);
    if (disp->mmap_data == MAP_FAILED)
    {
        printf("failed to mmap dmabuf: %s\n", strerror(errno));
        return -1;
    }

    uint8_t *data = NULL;
    ret = posix_memalign((void **)&data, sysconf(_SC_PAGESIZE), disp->len);
    if (ret)
    {
        printf("failed to allocate memory\n");
        return -1;
    }
    disp->data = data;

    // write data to output file
    uint8_t *pdata = NULL;
    pdata = malloc(data_len);
    if (!pdata)
    {
        printf("failed to allocate data\n");
        if (fp)
            fclose(fp);
        free(pdata);
        return -1;
    }

    struct netint_iocmd_issue_request uis;
    memset(&uis, 0, sizeof(uis));
    uis.fd = disp->fd;
    uis.data = disp->data;
    uis.len = disp->len;
    uis.dir = NI_DMABUF_READ_FROM_DEVICE;
    memset(disp->data, 0, disp->len);
    ret = ioctl(p_ctx->netint_fd, NETINT_IOCTL_ISSUE_REQ, &uis);
    if (ret < 0)
    {
        printf("failed to send req: %s\n", strerror(errno));
        if (fp)
            fclose(fp);
        free(pdata);
        return -1;
    }

    // poll to check if read complete
    struct pollfd pfds[1];
    pfds[0].fd = disp->fd;
    pfds[0].events = POLLOUT;
    pfds[0].revents = 0;
    ret = poll(pfds, 1, -1);
    if (ret < 0)
    {
        printf("failed to poll\n");
        if (fp)
            fclose(fp);
        free(pdata);
        return -1;
    }

    ni_log(NI_LOG_DEBUG, "data 0x%lx, data_len %u\n", (unsigned long)disp->data,
           data_len);
    memcpy(pdata, disp->data, data_len);

    if (fp)
    {
        if (0 !=
            write_dmabuf_data(pdata, fp, p_surface->ui16width,
                              p_surface->ui16height, format))
        {
            printf("failed to write file\n");
            if (fp)
                fclose(fp);
            free(pdata);
            return -1;
        }
        free(pdata);
    }

    return 0;
}
#endif

/*!*****************************************************************************
 *  \brief  Send decoder input data
 *
 *  \param
 *
 *  \return
 ******************************************************************************/
ni_retcode_t jpeg_decoder_send_data(ni_session_context_t *p_dec_ctx,
                               ni_session_data_io_t *p_in_data, int sos_flag,
                               int input_video_width, int input_video_height,
                               uint64_t pkt_size, unsigned long file_size,
                               unsigned long *total_bytes_sent, int print_time)
{
    uint64_t frame_pkt_size = pkt_size;
    int tx_size = 0;
    uint32_t send_size = 0;
    int new_packet = 0;
    int saved_prev_size = 0;
    ni_packet_t *p_in_pkt = &(p_in_data->data.packet);
    ni_retcode_t retval = NI_RETCODE_SUCCESS;

    ni_log(NI_LOG_DEBUG, "===> decoder_send_data <===\n");

    if (0 == p_in_pkt->data_len)
    {
        memset(p_in_pkt, 0, sizeof(ni_packet_t));
        ni_log(NI_LOG_DEBUG,
               "decoder_send_data * frame_pkt_size %" PRIu64 " pkt_size %" PRIu64 "\n",
               frame_pkt_size,
               pkt_size);
        p_in_pkt->p_data = NULL;
        p_in_pkt->data_len = frame_pkt_size;

        if (frame_pkt_size + p_dec_ctx->prev_size > 0)
        {
            ni_packet_buffer_alloc(p_in_pkt,
                                   (int)frame_pkt_size + p_dec_ctx->prev_size);
        }

        new_packet = 1;
        send_size = frame_pkt_size + p_dec_ctx->prev_size;
        saved_prev_size = p_dec_ctx->prev_size;
    } else
    {
        send_size = p_in_pkt->data_len;
    }

    p_in_pkt->start_of_stream = sos_flag;
    p_in_pkt->end_of_stream = 0;
    p_in_pkt->video_width = input_video_width;
    p_in_pkt->video_height = input_video_height;

    if (send_size == 0)
    {
        if (new_packet)
        {
            send_size =
                ni_packet_copy(p_in_pkt->p_data, g_curr_cache_pos, 0,
                               p_dec_ctx->p_leftover, &p_dec_ctx->prev_size);
        }
        p_in_pkt->data_len = send_size;
    } else
    {
        if (new_packet)
        {
            send_size =
                ni_packet_copy(p_in_pkt->p_data, g_curr_cache_pos, frame_pkt_size,
                               p_dec_ctx->p_leftover, &p_dec_ctx->prev_size);
            // p_in_pkt->data_len is the actual packet size to be sent to decoder
            p_in_pkt->data_len += saved_prev_size;
        }
    }

    tx_size =
        ni_device_session_write(p_dec_ctx, p_in_data, NI_DEVICE_TYPE_DECODER);

    if (tx_size < 0)
    {
        // Error
        fprintf(stderr, "Error: sending data error. rc:%d\n", tx_size);
        retval = NI_RETCODE_FAILURE;
        LRETURN;
    } else if (tx_size == 0)
    {
        ni_log(NI_LOG_DEBUG, "0 byte sent this time, sleep and will re-try.\n");
        ni_usleep(10000);
    }

    *total_bytes_sent += tx_size;

    if (tx_size > 0)
    {
        ni_log(NI_LOG_DEBUG, "decoder_send_data: reset packet_buffer.\n");
        ni_packet_buffer_free(p_in_pkt);
    }

    retval = NI_RETCODE_SUCCESS;

    END;

    return retval;
}

/*!*****************************************************************************
 *  \brief  Receive decoded output data from decoder
 *
 *  \param
 *
 *  \return 0: got YUV frame;  1: end-of-stream;  2: got nothing
 ******************************************************************************/
int jpeg_decoder_receive_data(ni_session_context_t *p_dec_ctx,
                         ni_session_data_io_t *p_out_data,
                         int output_video_width, int output_video_height,
                         FILE *p_file, unsigned long long *total_bytes_received,
                         int print_time, int write_to_file,
                         int * p_rx_size)
{
    int rc = NI_RETCODE_FAILURE;
    int end_flag = 0;
    int rx_size = 0;
    bool b_is_hwframe = p_dec_ctx->hw_action;
    ni_frame_t *p_out_frame = &(p_out_data->data.frame);
    int width, height;
    // for now read only planar data from decoder
    int is_planar = 1;

    ni_log(NI_LOG_DEBUG, "===> decoder_receive_data hwframe %d <===\n", b_is_hwframe);

    // prepare memory buffer for receiving decoded frame
    width = p_dec_ctx->actual_video_width > 0 ?
        (int)(p_dec_ctx->actual_video_width) :
        output_video_width;
    height = p_dec_ctx->active_video_height > 0 ?
        (int)(p_dec_ctx->active_video_height) :
        output_video_height;
    // allocate memory only after resolution is known (for buffer pool set up)
    int alloc_mem = (p_dec_ctx->active_video_width > 0 &&
                             p_dec_ctx->active_video_height > 0 ?
                         1 :
                         0);
    if (!b_is_hwframe)
    {
        rc = ni_decoder_frame_buffer_alloc(
            p_dec_ctx->dec_fme_buf_pool, &(p_out_data->data.frame), alloc_mem,
            width, height, p_dec_ctx->codec_format == NI_CODEC_FORMAT_H264,
            p_dec_ctx->bit_depth_factor, is_planar);
        if (NI_RETCODE_SUCCESS != rc)
        {
            LRETURN;
        }
        rx_size = ni_device_session_read(p_dec_ctx, p_out_data,
                                         NI_DEVICE_TYPE_DECODER);
    } else
    {
        rc = ni_frame_buffer_alloc(
            &(p_out_data->data.frame), width, height,
            p_dec_ctx->codec_format == NI_CODEC_FORMAT_H264, 1,
            p_dec_ctx->bit_depth_factor,
            3 /*3 is max supported hwframe output count per frame*/, is_planar);
        if (NI_RETCODE_SUCCESS != rc)
        {
            LRETURN;
        }
        rx_size = ni_device_session_read_hwdesc(p_dec_ctx, p_out_data,
                                                NI_DEVICE_TYPE_DECODER);
    }
    end_flag = p_out_frame->end_of_stream;

    if (rx_size < 0)
    {
        fprintf(stderr, "Error: receiving data error. rc:%d\n", rx_size);
        if (!b_is_hwframe)
        {
            ni_decoder_frame_buffer_free(&(p_out_data->data.frame));
        } else
        {
            ni_frame_buffer_free(&(p_out_data->data.frame));
        }
        rc = NI_RETCODE_FAILURE;
        LRETURN;
    } else if (rx_size > 0)
    {
        number_of_frames++;
        ni_log(NI_LOG_DEBUG, "Got frame # %" PRIu64 " bytes %d\n",
                       p_dec_ctx->frame_num, rx_size);

        ni_dec_retrieve_aux_data(p_out_frame);
    }

    // rx_size == 0 means no decoded frame is available now
    if (rx_size > 0 && p_file && write_to_file)
    {
        int i, j;
        for (i = 0; i < 3; i++)
        {
            uint8_t *src = p_out_frame->p_data[i];
            int plane_height = p_dec_ctx->active_video_height;
            int plane_width = p_dec_ctx->active_video_width;
            int write_height = output_video_height;
            int write_width = output_video_width;

            // plane_width is the actual Y stride size
            write_width *= p_dec_ctx->bit_depth_factor;

            if (i == 1 || i == 2)
            {
                plane_height /= 2;
                // U/V stride size is multiple of 128, following the calculation
                // in ni_decoder_frame_buffer_alloc
                plane_width = (((int)(p_dec_ctx->actual_video_width) / 2 *
                                    p_dec_ctx->bit_depth_factor +
                                127) /
                               128) *
                    128;
                write_height /= 2;
                write_width /= 2;
            }
            // apply the cropping window in writing out the YUV frame
            // for now the window is usually crop-left = crop-top = 0, and we
            // use this to simplify the cropping logic
            for (j = 0; j < plane_height; j++)
            {
                if (j < write_height &&
                    fwrite(src, write_width, 1, p_file) != 1)
                {
                    fprintf(stderr,
                            "Error: writing data plane %d: height %d error!\n",
                            i, plane_height);
                    fprintf(stderr, "Error: ferror rc = %d\n", ferror(p_file));
                }
                src += plane_width;
            }
        }
        if (fflush(p_file))
        {
            fprintf(stderr,
                    "Error: writing data frame flush failed! errno %d\n",
                    errno);
        }
    }

    *total_bytes_received += rx_size;
    *p_rx_size = rx_size;

    if (print_time)
    {
        ni_log(NI_LOG_INFO,"[R] Got:%d  Frames= %u  fps=%f  Total bytes %llu\n", rx_size,
               number_of_frames,
               ((float)number_of_frames /
                (float)(current_time.tv_sec - start_time.tv_sec)),
               (unsigned long long)*total_bytes_received);
    }

    if (end_flag)
    {
        printf("Decoder Receiving done.\n");
        rc = 1;
    } else if (0 == rx_size)
    {
        rc = 2;
    }

    ni_log(NI_LOG_DEBUG, "decoder_receive_data: success\n");

    END;

    return rc;
}


/*!*****************************************************************************
 *  \brief  decoder session open
 *
 *  \param
 *
 *  \return 0 if successful, < 0 otherwise
 ******************************************************************************/
int decoder_open_session(ni_session_context_t *p_dec_ctx, int iXcoderGUID,
                         ni_xcoder_params_t *p_dec_params)
{
    int ret = 0;

    p_dec_ctx->codec_format = NI_CODEC_FORMAT_JPEG;
    p_dec_ctx->p_session_config = p_dec_params;

    // assign the card GUID in the decoder context and let session open
    // take care of the rest
    p_dec_ctx->hw_id = iXcoderGUID;

    // default: little endian
    p_dec_ctx->src_bit_depth = 8;
    p_dec_ctx->src_endian = NI_FRAME_LITTLE_ENDIAN;
    p_dec_ctx->bit_depth_factor = 1;

    if (p_dec_params->dec_input_params.hwframes)
    {
        p_dec_ctx->hw_action = NI_CODEC_HW_ENABLE;
    } else
    {
        p_dec_ctx->hw_action = NI_CODEC_HW_NONE;
    }

    ret = ni_device_session_open(p_dec_ctx, NI_DEVICE_TYPE_DECODER);

    if (ret < 0)
    {
        fprintf(stderr, "Error: ni_decoder_session_open() failure!\n");
        return -1;
    } else
    {
        printf("Decoder device %d session open successful.\n", iXcoderGUID);
        return 0;
    }
}

void print_usage(void)
{
    printf("JPEG decoding application using NETINT Quadra\n"
           "Libxcoder release v%s\n"
           "Usage: ni_jpeg_test [options]\n"
           "\n"
           "options:\n"
           "-h | --help             Show help.\n"
           "-v | --version          Print version info.\n"
           "-l | --loglevel         Set loglevel of libxcoder API.\n"
           "                        [none, fatal, error, info, debug, trace]\n"
           "                        (Default: info)\n"
           "-c | --card             Set card index to use.\n"
           "                        See `ni_rsrc_mon` for cards on system.\n"
           "                        (Default: 0)\n"
           "-i | --input            Text file with list of paths of jpeg files.\n"
           "                        ex: test_3840x2160_1920x1080_file.txt\n"
           "-d | --decoder_params   Decoding params. See \"Decoding "
           "Parameters\" chapter in\n"
           "                        "
           "QuadraIntegration&ProgrammingGuide*.pdf for help.\n"
           "                        (Default: \"\")\n"
           "-o | --output           Output file path. eg:-o out.yuv. The output format is the same as -f.\n"
           "-f | --scaler_params    Scaling params. ex: "
           "scale_width=640:scale_height=360:pad_width=640:pad_height=384:format=0:p2p=0. "
           "{format [ 0:yuv420p  1:rgba  2:rgbp ]  p2p [ 0:no p2p  1:use p2p ]}\n",
           NI_XCODER_REVISION);
}

static int ni_scaler_params_set_value(ni_filter_params_t *params,
                                      const char *name, const char *value)
{
    if (!params)
    {
        printf("Error: Null pointer parameters passed\n");
        return -1;
    }

    if (!name)
    {
        printf("Error: Null name pointer parameters passed\n");
        return -1;
    }

    if (!strcmp("scale_width", name))
    {
        params->scale_width = atoi(value);
    } else if (!strcmp("scale_height", name))
    {
        params->scale_height = atoi(value);
    }  else if (!strcmp("pad_width", name))
    {
        params->pad_width = atoi(value);
    } else if (!strcmp("pad_height", name))
    {
        params->pad_height = atoi(value);
    } else if (!strcmp("p2p", name))
    {
        int is_p2p = atoi(value);
        params->p2p = is_p2p;
    } else if (!strcmp("format", name))
    {
        int format = atoi(value);
        if (format == 0)
        {
            params->format = GC620_I420;
        } else if (format == 1)
        {
            params->format = GC620_RGBA8888;
        } else if (format == 2)
        {
            params->format = GC620_RGB888_PLANAR;
        } else
        {
            printf("Error: invalid scaler parameters\n");
            return -1;
        }
    } else
    {
        printf("Error: invalid scaler parameters\n");
        return -1;
    }
    return 0;
}

static int retrieve_filter_params(char xcoderParams[],
                                  ni_filter_params_t *params)
{
    char key[64], value[64];
    char *curr = xcoderParams, *colon_pos;
    int ret = 0;

    while (*curr)
    {
        colon_pos = strchr(curr, ':');

        if (colon_pos)
        {
            *colon_pos = '\0';
        }

        if (strlen(curr) > sizeof(key) + sizeof(value) - 1 ||
            ni_param_get_key_value(curr, key, value))
        {
            fprintf(stderr,
                    "Error: scaler-params key/value not "
                    "retrieved: %s\n",
                    curr);
            ret = -1;
            break;
        }
        ret = ni_scaler_params_set_value(params, key, value);
        if (ret != 0)
        {
            printf("Error: failed to parse the input scaler parameters\n");
            ret = -1;
            break;
        }

        if (colon_pos)
        {
            curr = colon_pos + 1;
        } else
        {
            curr += strlen(curr);
        }
    }
    return ret;
}

/*!*****************************************************************************
 *  \brief  Init scaler params here - both user setting params and fixed params.
 *
 *  \param
 *
 *  \return 0 if successful, < 0 otherwise
 ******************************************************************************/
void init_scaler_params(ni_scaler_input_params_t *p_scaler_params,
                        ni_scaler_opcode_t op, int in_rec_width,
                        int in_rec_height, int in_rec_x, int in_rec_y,
                        int out_rec_x, int out_rec_y)
{
    p_scaler_params->op = op;
    // input_format/width/height, output_format/width/height should be assigned by users for all ops
    if (op == NI_SCALER_OPCODE_CROP)
    {
        // fixed numbers
        p_scaler_params->out_rec_width = 0;
        p_scaler_params->out_rec_height = 0;
        p_scaler_params->out_rec_x = 0;
        p_scaler_params->out_rec_y = 0;
        p_scaler_params->rgba_color = 0;

        // params set by user
        p_scaler_params->in_rec_width = in_rec_width;
        p_scaler_params->in_rec_height = in_rec_height;
        p_scaler_params->in_rec_x = in_rec_x;
        p_scaler_params->in_rec_y = in_rec_y;
    } else if (op == NI_SCALER_OPCODE_SCALE)
    {
        // fixed params
        p_scaler_params->in_rec_width = 0;
        p_scaler_params->in_rec_height = 0;
        p_scaler_params->in_rec_x = 0;
        p_scaler_params->in_rec_y = 0;

        p_scaler_params->out_rec_width = 0;
        p_scaler_params->out_rec_height = 0;
        p_scaler_params->out_rec_x = 0;
        p_scaler_params->out_rec_y = 0;

        p_scaler_params->rgba_color = 0;
    } else if (op == NI_SCALER_OPCODE_PAD)
    {
        // fixed params
        p_scaler_params->in_rec_width = p_scaler_params->input_width;
        p_scaler_params->in_rec_height = p_scaler_params->input_height;
        p_scaler_params->in_rec_x = 0;
        p_scaler_params->in_rec_y = 0;

        p_scaler_params->out_rec_width = p_scaler_params->input_width;
        p_scaler_params->out_rec_height = p_scaler_params->input_height;

        /*
            Scaler uses BGRA color, or ARGB in little-endian
            ui32RgbaColor = (s->rgba_color[3] << 24) | (s->rgba_color[0] << 16) |
                            (s->rgba_color[1] << 8) | s->rgba_color[2];
            here p_scaler_params->rgba_color = ui32RgbaColor;
        */
        p_scaler_params->rgba_color =
            4278190080;   // now padding color is black

        // params set by user
        p_scaler_params->out_rec_x = out_rec_x;
        p_scaler_params->out_rec_y = out_rec_y;
    } else if (op == NI_SCALER_OPCODE_OVERLAY)
    {
        // fixed params
        // set the in_rec params to the w/h of overlay(the upper) frames
        p_scaler_params->in_rec_width = p_scaler_params->input_width;
        p_scaler_params->in_rec_height = p_scaler_params->input_height;

        // the output w/h is the main frame's w/h (main frame is the lower/background frame)
        p_scaler_params->out_rec_width = p_scaler_params->output_width;
        p_scaler_params->out_rec_height = p_scaler_params->output_height;
        p_scaler_params->out_rec_x = 0;
        p_scaler_params->out_rec_y = 0;
        p_scaler_params->rgba_color = 0;

        // params set by user
        p_scaler_params->in_rec_x = in_rec_x;
        p_scaler_params->in_rec_y = in_rec_x;
    }
}

/*!*****************************************************************************
 *  \brief  open scaler session
 *
 *  \param
 *
 *  \return 0 if successful, < 0 otherwise
 ******************************************************************************/
int scaler_session_open(ni_session_context_t *p_scaler_ctx, int iXcoderGUID,
                        ni_scaler_opcode_t op)
{
    int ret = 0;

    p_scaler_ctx->session_id = NI_INVALID_SESSION_ID;

    p_scaler_ctx->device_handle = NI_INVALID_DEVICE_HANDLE;
    p_scaler_ctx->blk_io_handle = NI_INVALID_DEVICE_HANDLE;
    p_scaler_ctx->hw_id = iXcoderGUID;
    p_scaler_ctx->device_type = NI_DEVICE_TYPE_SCALER;
    p_scaler_ctx->scaler_operation = op;
    p_scaler_ctx->keep_alive_timeout = NI_DEFAULT_KEEP_ALIVE_TIMEOUT;

    ret = ni_device_session_open(p_scaler_ctx, NI_DEVICE_TYPE_SCALER);

    if (ret < 0)
    {
        fprintf(stderr, "Error: ni_scaler_session_open() failure!\n");
        return -1;
    } else
    {
        ni_log(NI_LOG_DEBUG,"Scaler session open: device_handle %d, session_id %u.\n",
               p_scaler_ctx->device_handle, p_scaler_ctx->session_id);
        return 0;
    }
}

/*!*****************************************************************************
 *  \brief  Launch scaler operation and get the result hw frame
 *
 *  \param
 *
 *  \return 0 if successful, < 0 otherwise
 ******************************************************************************/
int launch_scaler_operation(ni_session_context_t *p_ctx,
                            ni_frame_t *p_frame_in_up,
                            ni_session_data_io_t *p_data_out,
                            ni_scaler_input_params_t scaler_params,
                            int iXcoderGUID)
{
    int ret = 0;
    if (p_ctx->session_id == NI_INVALID_SESSION_ID)
    {
        // Open scaler session
        if (0 != scaler_session_open(p_ctx, iXcoderGUID, scaler_params.op))
        {
            fprintf(stderr, "Error: scaler open session error\n");
            return -1;
        }

        // init scaler hwframe pool
        if (0 != ni_scaler_frame_pool_alloc(p_ctx, scaler_params))
        {
            fprintf(stderr, "Error: init filter hwframe pool\n");
            return -1;
        }
    }

    // allocate a ni_frame_t structure on the host PC
    if (0 !=
        ni_frame_buffer_alloc_hwenc(&(p_data_out->data.frame),
                                    scaler_params.output_width,
                                    scaler_params.output_height, 0))
    {
        return -1;
    }
    niFrameSurface1_t *frame_surface_up;
    frame_surface_up = (niFrameSurface1_t *)((
        p_frame_in_up->p_data[3]));   // out_frame retrieved from decoder

    // allocate scaler input frame
    if (0 !=
        ni_scaler_input_frame_alloc(p_ctx, scaler_params, frame_surface_up))
    {
        return -1;
    }
    // Allocate scaler destination frame.
    if (0 != ni_scaler_dest_frame_alloc(p_ctx, scaler_params, frame_surface_up))
    {
        return -1;
    }

    // Retrieve hardware frame info from 2D engine and put it in the ni_frame_t structure.
    ret =
        ni_device_session_read_hwdesc(p_ctx, p_data_out, NI_DEVICE_TYPE_SCALER);
    if (ret < 0)
    {
        ni_frame_buffer_free(p_frame_in_up);
        ni_frame_buffer_free(&(p_data_out->data.frame));
    }
    return ret;
}


/*!*****************************************************************************
 *  \brief  Do a scale and/or format-change operation.
 *
 *  \param
 *
 *  \return 0 if successful, < 0 otherwise
 ******************************************************************************/
int scale_filter(ni_session_context_t *p_ctx, ni_frame_t *p_frame_in,
                 ni_session_data_io_t *p_data_out, int scale_width,
                 int scale_height, int out_format, int iXcoderGUID)
{
    niFrameSurface1_t *p_surface_in;
    p_surface_in = (niFrameSurface1_t *)(p_frame_in->p_data[3]);

    ni_scaler_input_params_t scale_params;
    scale_params.output_format = out_format;   // rgba or bgrp or yuv420p;
    scale_params.output_width = scale_width;
    scale_params.output_height = scale_height;
    scale_params.input_format =
        GC620_I420;   // now assume the decoded frame format is GC620_I420;
    scale_params.input_width = p_surface_in->ui16width;
    scale_params.input_height = p_surface_in->ui16height;
    init_scaler_params(&scale_params, NI_SCALER_OPCODE_SCALE, 0, 0, 0, 0, 0, 0);
    int ret = launch_scaler_operation(p_ctx, p_frame_in, p_data_out,
                                      scale_params, iXcoderGUID);
    if (ret != 0)
    {
        printf("Failed to lauch scaler operation %d\n", scale_params.op);
        return -1;
    }
    return 0;
}

/*!*****************************************************************************
 *  \brief  Do a scale and/or format-change operation.
 *
 *  \param
 *
 *  \return 0 if successful, < 0 otherwise
 ******************************************************************************/
int pad_filter( ni_session_context_t *p_pad_ctx,
                ni_session_context_t *p_fmt_ctx, ni_frame_t *p_frame_in,
                ni_session_data_io_t *p_data_out, int pad_width,
                int pad_height, int out_format, int iXcoderGUID)
{
    int ret;
    niFrameSurface1_t *pad_frame_surface;
    pad_frame_surface = (niFrameSurface1_t *)(p_frame_in->p_data[3]);

    ni_scaler_input_params_t pad_params = {0};
    pad_params.output_format = GC620_I420;   // yuv420p;
    pad_params.output_width = pad_width;
    pad_params.output_height = pad_height;
    pad_params.input_format =
        GC620_I420;   // now assume the decoded frame format is GC620_I420;
    pad_params.input_width = pad_frame_surface->ui16width;
    pad_params.input_height = pad_frame_surface->ui16height;

    init_scaler_params(&pad_params, NI_SCALER_OPCODE_PAD, 0, 0, 0, 0,
                       (pad_width - pad_params.input_width) / 2,
                       (pad_height - pad_params.input_height) / 2);
    ni_session_data_io_t pad_data = {0};
    if (out_format == GC620_I420)
    {
        ret = launch_scaler_operation(p_pad_ctx, p_frame_in,
                                      p_data_out, pad_params, iXcoderGUID);
    }
    else
    {
        ret = launch_scaler_operation(p_pad_ctx, p_frame_in,
                                      &pad_data, pad_params, iXcoderGUID);
    }

    if (ret != 0)
    {
        printf("Failed to launch scaler operation %d\n", pad_params.op);
        return -1;
    }

    if (out_format != GC620_I420)   // use scale filter to do format change
    {
        niFrameSurface1_t *fmt_frame_surface =
            (niFrameSurface1_t *)(pad_data.data.frame.p_data[3]);
        fmt_frame_surface->ui16width = pad_params.output_width;
        fmt_frame_surface->ui16height = pad_params.output_height;
        ret = scale_filter(p_fmt_ctx, &(pad_data.data.frame), p_data_out,
                           pad_params.output_width,
                           pad_params.output_height, out_format, iXcoderGUID);
        ni_hwframe_buffer_recycle(fmt_frame_surface,
                                  fmt_frame_surface->device_handle);
        ni_frame_buffer_free(&(pad_data.data.frame));
        if (ret != 0)
        {
            printf("Failed to launch scaler operation 0\n");
            return -1;
        }
    }

    return 0;
}

/*!*****************************************************************************
 *  \brief  Write scaler p2p or hwdl output.
 *
 *  \param
 *
 *  \return 0 if successful, < 0 otherwise
 ******************************************************************************/
int scaler_output_write(ni_session_context_t *p_ctx,
                        ni_session_data_io_t *p_data_out, int out_width,
                        int out_height, int out_format, FILE *fp,
                        disp_buffer_t *disp)
{
    if (!fp)
        return 0;
    niFrameSurface1_t *p_frame_surface =
        (niFrameSurface1_t *)(p_data_out->data.frame.p_data[3]);
    int ret;

    if (p_ctx->isP2P)
    {
#ifndef _WIN32
        ret = read_from_dmabuf(p_ctx, p_frame_surface, disp, out_format, fp);
        if (ret != 0)
        {
            ni_frame_buffer_free(&(p_data_out->data.frame));
            return -1;
        }
#endif
    } else
    {
        // use hwdownload to download hw frame, recycle hwframe buffer
        ni_session_data_io_t hwdl_session_data = {0};
        ret = hwdl_frame(p_ctx, &hwdl_session_data, &(p_data_out->data.frame),
                        gc620_to_ni_pix_fmt(out_format));
        if (ret <= 0)
        {
            ni_frame_buffer_free(&(p_data_out->data.frame));
            return -1;
        }

        // write hwdl frame to output_file
        write_rawvideo_data(fp, out_width, out_height, gc620_to_ni_pix_fmt(out_format),
                            &(hwdl_session_data.data.frame));
        ni_frame_buffer_free(&(hwdl_session_data.data.frame));
    }
    return 0;
}

/*!*****************************************************************************
 *  \brief  get picture size.
 *
 *  \param
 *
 *  \return 0 if successful, < 0 otherwise
 ******************************************************************************/
void get_jpeg_size(int *jpgHeight, int *jpgWidth)
{
    unsigned char HeightH;
    unsigned char HeightL;
    unsigned char WidthH;
    unsigned char WidthL;
    unsigned char BitDepth;
    int SOF_find=0;
    while (!SOF_find)
    {
        if (*g_curr_cache_pos != 0xFF)
        {
            g_curr_cache_pos++;
            continue;
        }
        //find marker
        g_curr_cache_pos++;
        switch (*g_curr_cache_pos)
        {
            case APP0:
                 break;
            case SOF0:
                g_curr_cache_pos += 3;
                BitDepth = *g_curr_cache_pos;
                g_curr_cache_pos++;
                HeightH = *g_curr_cache_pos;
                HeightL = *(g_curr_cache_pos + 1);
                WidthH = *(g_curr_cache_pos + 2);
                WidthL = *(g_curr_cache_pos + 3);
                if (HeightH == 0x00 || WidthH == 0x00)
                    break;
                if (BitDepth != 0x08)  //The demo only support 8 bit.
                {
                    ni_log(NI_LOG_ERROR, "The demo does not support %x bit depth, exit!\n",BitDepth);
                    exit(-1);
                }

                SOF_find = 1; //get picture width and height
                break;
            default:
                break;
        }
    }

    *jpgHeight = HeightH * 256 + HeightL;
    *jpgWidth = WidthH * 256 + WidthL;
    ni_log(NI_LOG_DEBUG, "BitDepth: %x, jpeg dimensions: %dx%d\n", BitDepth, *jpgWidth, *jpgHeight);
    g_curr_cache_pos = g_file_cache;
}

/*!*****************************************************************************
 *  \brief  read one jpeg file on a line in the input file
 *
 *  \param
 *
 *  \return 0 if successful, < 0 otherwise
 ******************************************************************************/
static uint64_t read_file(FILE *fopentxt)
{
    char tmpfile[256] = {0};
    char curfile[256] = {0};

    if (fgets(tmpfile, sizeof(tmpfile), fopentxt) == NULL)
    {
        ni_log(NI_LOG_ERROR, "Read error or end of %s.\n", filename);
        return -1;
    }
    memcpy(curfile, tmpfile, strlen(tmpfile) - 1);

    pfs = open(curfile, O_RDONLY);
    if (pfs == -1)
    {
        ni_log(NI_LOG_ERROR, "The input file open fail %s\n", curfile);
        return -1;
    }
    lseek(pfs, 0, SEEK_END);
    total_file_size = lseek(pfs, 0, SEEK_CUR);
    lseek(pfs, 0, SEEK_SET);

    uint32_t read_chunk = 4096;
    uint64_t tmpFileSize = total_file_size;

    //try to allocate memory for input file buffer, quit if failure
    if (total_file_size > 0 && !(g_file_cache = malloc(total_file_size)))
    {
        fprintf(stderr,
                "Error: input file %s size %llu exceeding max malloc, quit\n",
                curfile, (unsigned long long)total_file_size);
        return -1;
    }
    g_curr_cache_pos = g_file_cache;
    ni_log(NI_LOG_DEBUG, "Reading %llu bytes in total from file %s\n",
            (unsigned long long)total_file_size, curfile);

    while (tmpFileSize)
    {
        if (read_chunk > tmpFileSize)
        {
            read_chunk = tmpFileSize;
        }
        int one_read_size = read(pfs, g_curr_cache_pos, read_chunk);
        if (one_read_size == -1)
        {
            fprintf(stderr, "Error: reading file, quit! left-to-read %llu\n",
                    (unsigned long long)tmpFileSize);
            fprintf(stderr, "Error: input file %s read error\n",curfile);
            return -1;
        } else
        {
            tmpFileSize -= one_read_size;
            g_curr_cache_pos += one_read_size;
        }
    }
    ni_log(NI_LOG_DEBUG, "read %llu bytes from input file %s into memory\n",
            (unsigned long long)total_file_size, curfile);
    g_curr_cache_pos = g_file_cache;
    return total_file_size;
}

int main(int argc, char *argv[])
{
    int sos_flag = 0;
    int iXcoderGUID = 0;
    unsigned long total_bytes_sent;
    unsigned long long total_bytes_received;
    FILE *p_file = NULL;
    int input_video_width;
    int input_video_height;
    int output_video_width ;
    int output_video_height;
    uint64_t pkt_size;
    ni_xcoder_params_t *p_dec_api_param = NULL;
    char decConfXcoderParams[2048] = {0};
    char scaConfXcoderParams[2048] = {0};
    int ret = 0;
    ni_log_level_t log_level = NI_LOG_DEBUG;
    decod_status_t decode_status;

    // Input arg handling
    int opt;
    int opt_index;
    const char *opt_string = "hvl:c:i:d:f:o:";
    static struct option long_options[] = {
        {"help", no_argument, NULL, 'h'},
        {"version", no_argument, NULL, 'v'},
        {"loglevel", required_argument, NULL, 'l'},
        {"card", required_argument, NULL, 'c'},
        {"input", required_argument, NULL, 'i'},
        {"decoder_params", required_argument, NULL, 'd'},
        {"scaler_params", required_argument, NULL, 'f'},
        {"output", required_argument, NULL, 'o'},
        {NULL, 0, NULL, 0},
    };

    while ((opt = getopt_long(argc, argv, opt_string, long_options,
                              &opt_index)) != -1)
    {
        switch (opt)
        {
            case 'h':
                print_usage();
                exit(0);
            case 'v':
                printf("Release ver: %s\n"
                       "API ver:     %s\n"
                       "Date:        %s\n"
                       "ID:          %s\n",
                       NI_XCODER_REVISION, LIBXCODER_API_VERSION,
                       NI_SW_RELEASE_TIME, NI_SW_RELEASE_ID);
                exit(0);
            case 'l':
                log_level = arg_to_ni_log_level(optarg);
                if (log_level != NI_LOG_INVALID)
                {
                    ni_log_set_level(log_level);
                } else {
                    fprintf(stderr, "-l | --loglevel %s", optarg);
                }
                break;
            case 'c':
                iXcoderGUID = atoi(optarg);
                break;
            case 'i':
                filename = optarg;
                break;
            case 'd':
                strcpy(decConfXcoderParams, optarg);
                break;
            case 'f':
                strcpy(scaConfXcoderParams, optarg);
                break;
            case 'o':
                outputfile = optarg;
                break;
            default:
                print_usage();
                exit(1);
        }
    }
    // Check required args are present
    if (!filename)
    {
        printf("Error: missing argument for -i | --input\n");
        exit(-1);
    }

    //open input file text
    FILE *fopentxt = fopen(filename, "r");
    if (!fopentxt)
    {
        printf("open file %s fail!\n", filename);
        exit(-1);
    }

    //open output file
    if (strcmp(outputfile, "null") != 0 && strcmp(outputfile, "/dev/null") != 0)
    {
        p_file = fopen(outputfile, "wb");
        if (p_file == NULL)
        {
            fprintf(stderr, "Error: cannot open %s\n", outputfile);
            goto end;
        }
        ni_log(NI_LOG_DEBUG, "SUCCESS: Opened output file: %s\n", outputfile);
    }
    else
    {
        ni_log(NI_LOG_DEBUG, "SUCCESS: output file %s implies dropping output\n", outputfile);
    }

    p_dec_api_param = malloc(sizeof(ni_xcoder_params_t));
    if (!p_dec_api_param)
    {
        printf("Error: failed to allocate p_dec_api_param\n");
        return -1;
    }

    //open one picture file in the input file
    int write_frame = 0,frames_to_decode = 0;
    int rx_size;
    pkt_size = read_file(fopentxt);
    if(pkt_size == -1)
    {
        ni_log(NI_LOG_ERROR, "first picture file open failed,exit\n");
        return -1;
    }
    else{
        frames_to_decode++;
    }
    get_jpeg_size(&input_video_height, &input_video_width);
    output_video_width = input_video_width;
    output_video_height = input_video_height;
    decode_status = DECODE_START;

    // set up decoder p_config with some hard coded numbers
    if (ni_decoder_init_default_params(p_dec_api_param, 25, 1, 0, input_video_width,
                                       input_video_height) < 0)
    {
        fprintf(stderr, "Error: decoder p_config set up error\n");
        return -1;
    }

    ni_session_context_t dec_ctx = {0};
    ni_session_context_t sca_ctx = {0};
    ni_session_context_t pad_ctx = {0};
    ni_session_context_t fmt_ctx = {0};

    if (ni_device_session_context_init(&dec_ctx) < 0)
    {
        fprintf(stderr, "Error: init decoder context error\n");
        return -1;
    }

    if (ni_device_session_context_init(&sca_ctx) < 0)
    {
        fprintf(stderr, "Error: init decoder context error\n");
        return -1;
    }

    if (ni_device_session_context_init(&pad_ctx) < 0)
    {
        fprintf(stderr, "Error: init decoder context error\n");
        return -1;
    }

    if (ni_device_session_context_init(&fmt_ctx) < 0)
    {
        fprintf(stderr, "Error: init decoder context error\n");
        return -1;
    }

    dec_ctx.p_session_config = NULL;
    dec_ctx.session_id = NI_INVALID_SESSION_ID;
    dec_ctx.codec_format = NI_CODEC_FORMAT_JPEG;

    // assign the card GUID in the decoder context and let session open
    // take care of the rest
    dec_ctx.device_handle = dec_ctx.blk_io_handle =
        NI_INVALID_DEVICE_HANDLE;
    dec_ctx.hw_id = iXcoderGUID;
    dec_ctx.p_session_config = p_dec_api_param;
    // default: little endian
    dec_ctx.src_bit_depth = 8;
    dec_ctx.src_endian = NI_FRAME_LITTLE_ENDIAN;
    dec_ctx.bit_depth_factor = 1;

    // check and set ni_decoder_params from --decoder_params
    if (ni_retrieve_decoder_params(decConfXcoderParams, p_dec_api_param,
                                &dec_ctx))
    {
        fprintf(stderr, "Error: decoder p_config parsing error\n");
        return -1;
    }

    // Decode, use all the parameters specified by user
    if (0 !=
        (ret = decoder_open_session(&dec_ctx, iXcoderGUID, p_dec_api_param)))
    {
        goto end;
    }

    sos_flag = 1;
    total_bytes_received = 0;
    total_bytes_sent = 0;

    ni_log(NI_LOG_DEBUG, "Scaling Mode\n");
    // default filter params
    ni_filter_params_t filter_params = {0};
    filter_params.scale_width = 1280;
    filter_params.scale_height = 720;
    filter_params.pad_width = 1280;
    filter_params.pad_height = 720;
    filter_params.format = GC620_I420;

    if (0 !=
        (ret = retrieve_filter_params(scaConfXcoderParams, &filter_params)))
    {
        return -1;
    }
    int is_p2p = filter_params.p2p;
    int scale_width = filter_params.scale_width;
    int scale_height = filter_params.scale_height;
    int pad_width = filter_params.pad_width;
    int pad_height = filter_params.pad_height;
    int out_format = filter_params.format;
    if (is_p2p)
    {
        sca_ctx.isP2P = 1;
        if (out_format != GC620_I420)
            fmt_ctx.isP2P = 1;
        else
            pad_ctx.isP2P = 1;
    }

    niFrameSurface1_t dec_hwframe_tracker = {0};
    niFrameSurface1_t scaler_hwframe_tracker = {0};
    niFrameSurface1_t pad_hwframe_tracker = {0};
    disp_buffer_t *disp1 = NULL;
    disp_buffer_t *disp2 = NULL;
    if (is_p2p)
    {
        disp1 = calloc(1, sizeof(disp_buffer_t));
        disp2 = calloc(1, sizeof(disp_buffer_t));
    }

    ni_session_data_io_t in_pkt = {0};
    ni_session_data_io_t out_frame = {0};

    /* this test the scale before pad output correct or not.
    FILE *p_scale_out = NULL;
    if (p_file)
    {
        char *scale_outfile;
        char *p = strtok(outputfile, ".");
        if (!p)
        {
            scale_outfile = strcat(outputfile, "_scale");
        }
        while (p)
        {
            char *name = (char *)malloc(strlen(p));
            strcpy(name, p);
            strcat(name, "_scale.");
            p = strtok(NULL, ".");
            scale_outfile = strcat(name, p);
            break;
        }
        p_scale_out = fopen(scale_outfile, "wb");
    }
    */

    send_fin_flag = 0;
    receive_fin_flag = 0;
    (void)ni_gettimeofday(&start_time, NULL);
    (void)ni_gettimeofday(&previous_time, NULL);
    (void)ni_gettimeofday(&current_time, NULL);

    while ((decode_status != DECODE_END) || (frames_to_decode != write_frame))
    {
        (void)ni_gettimeofday(&current_time, NULL);
        int print_time = ((current_time.tv_sec - previous_time.tv_sec) > 1);

        if (decode_status == DECODE_START)
        {
            // Sending
            send_fin_flag = jpeg_decoder_send_data(
                &dec_ctx, &in_pkt, sos_flag, input_video_width,
                input_video_height, pkt_size, total_file_size,
                &total_bytes_sent, print_time);

            sos_flag = 0;
            if (send_fin_flag < 0)
            {
                fprintf(stderr, "Error: decoder_send_data() failed, rc: %d\n",
                        send_fin_flag);
                goto end;
            }
            if (total_bytes_sent == total_file_size)
                decode_status = DECODE_READ;
        }

        if (decode_status == DECODE_READ)
        {
            total_bytes_sent = 0;
            free(g_file_cache);
            g_file_cache = NULL;
            close(pfs);
            pkt_size = read_file(fopentxt);
            if (pkt_size == -1)
            {
                decode_status = DECODE_END;
            }
            else {
                frames_to_decode++;
                get_jpeg_size(&input_video_height, &input_video_width);
                decode_status = DECODE_WAIT;
            }
        }

        // Receiving
        receive_fin_flag = jpeg_decoder_receive_data(
            &dec_ctx, &out_frame, output_video_width, output_video_height,
            p_file, &total_bytes_received, print_time, 0,&rx_size);   // not write to file
        if (print_time)
        {
            previous_time = current_time;
        }

        if (receive_fin_flag < 0 || out_frame.data.frame.end_of_stream)
        {
            ni_frame_buffer_free(&(out_frame.data.frame));
            goto end;
        } else if (receive_fin_flag != 2)
        {
            // got yuv hwframe from decoder. save dec hwframe idx
            memcpy(&dec_hwframe_tracker,
                (uint8_t *)out_frame.data.frame.p_data[3],
                sizeof(niFrameSurface1_t));
            niFrameSurface1_t *decoded_surface =
                (niFrameSurface1_t *)(out_frame.data.frame.p_data[3]);

            // set decoded frame params
            decoded_surface->ui16width = output_video_width;
            decoded_surface->ui16height = output_video_height;
            decoded_surface->bit_depth = 1;
            decoded_surface->encoding_type = NI_PIXEL_PLANAR_FORMAT_PLANAR;

            // scale the decoded frame(always yuv) and write to output file YUV420P
            ni_session_data_io_t scale_session_data = {0};
            ret = scale_filter(&sca_ctx, &(out_frame.data.frame),
                            &scale_session_data, scale_width,
                            scale_height, GC620_I420, iXcoderGUID);
            // free decoded hw frame
            if (dec_hwframe_tracker.ui16FrameIdx)
            {
                ni_hwframe_buffer_recycle(
                    &dec_hwframe_tracker,
                    dec_hwframe_tracker.device_handle);
                dec_hwframe_tracker.ui16FrameIdx = 0;
            }
            if (ret != 0)
                goto end;

            //Save scaler HwFrameIdx
            memcpy(&scaler_hwframe_tracker,
                (uint8_t *)(scale_session_data.data.frame.p_data[3]),
                sizeof(niFrameSurface1_t));

            //this test the scale before pad output correct or not.
            // ret = scaler_output_write(&sca_ctx, &scale_session_data,
            //                         scale_width, scale_height, GC620_I420,
            //                         p_scale_out, disp1);

            niFrameSurface1_t *p_scale_surface =
                (niFrameSurface1_t *)(scale_session_data.data.frame.p_data[3]);
            p_scale_surface->ui16width = scale_width;
            p_scale_surface->ui16height = scale_height;
            p_scale_surface->bit_depth = 1;
            p_scale_surface->encoding_type = NI_PIXEL_PLANAR_FORMAT_PLANAR;
            if (ret != 0)
                goto end;

            // pad the decoded frame(always yuv) and write to output file
            ni_session_data_io_t pad_session_data = {0};
            ret = pad_filter(&pad_ctx, &fmt_ctx,&(scale_session_data.data.frame),
                            &pad_session_data, pad_width,
                            pad_height, out_format, iXcoderGUID);

            if (scaler_hwframe_tracker.ui16FrameIdx)
            {
                ni_hwframe_buffer_recycle(
                    &scaler_hwframe_tracker,
                    scaler_hwframe_tracker.device_handle);
                scaler_hwframe_tracker.ui16FrameIdx = 0;
            }
            if (ret != 0)
                goto end;

            // Save scaler HwFrameIdx
            memcpy(&pad_hwframe_tracker,
                (uint8_t *)(pad_session_data.data.frame.p_data[3]),
                sizeof(niFrameSurface1_t));

            niFrameSurface1_t *p_pad_surface =
                (niFrameSurface1_t *)(pad_session_data.data.frame.p_data[3]);
            p_pad_surface->ui16width = pad_width;
            p_pad_surface->ui16height = pad_height;
            p_pad_surface->bit_depth = 1;
            p_pad_surface->encoding_type = NI_PIXEL_PLANAR_FORMAT_PLANAR;
            if(out_format == GC620_I420)
                ret = scaler_output_write(&pad_ctx, &pad_session_data,
                                        pad_width, pad_height, out_format,
                                        p_file, disp2);
            else
                ret = scaler_output_write(&fmt_ctx, &pad_session_data,
                                        pad_width, pad_height, out_format,
                                        p_file, disp2);
            if (ret != 0)
                goto end;
            if (pad_hwframe_tracker.ui16FrameIdx)
            {
                ni_hwframe_buffer_recycle(
                    &pad_hwframe_tracker,
                    pad_hwframe_tracker.device_handle);
                pad_hwframe_tracker.ui16FrameIdx = 0;
            }

            // free decoded frame and scaler pad output frame
            ni_frame_buffer_free(&(out_frame.data.frame));
            ni_frame_buffer_free(&(scale_session_data.data.frame));
            ni_frame_buffer_free(&(pad_session_data.data.frame));
            write_frame++;
            if (decode_status != DECODE_END)
                decode_status = DECODE_START;

            //for next picture output size
            output_video_width = input_video_width;
            output_video_height = input_video_height;
        } else
        {
            ni_frame_buffer_free(&(out_frame.data.frame));
        }
    }
    unsigned int time_diff =
        (unsigned int)(current_time.tv_sec - start_time.tv_sec);
    if (time_diff == 0)
        time_diff = 1;

    ni_log(NI_LOG_INFO,"[R] Got:  Frames= %u  fps=%f  Total bytes %llu\n",
            number_of_frames, ((float)number_of_frames / (float)time_diff),
            total_bytes_received);

    // close device
    if (is_p2p)
    {
        if (disp1)
        {
            if (disp1->fd >= 0)
            {
                if (disp1->mmap_data != MAP_FAILED)
                {
                    munmap((void *)disp1->mmap_data, disp1->len);
                }
                close(disp1->fd);
                free(disp1->data);
            }
            free(disp1);
            disp1 = NULL;
        }
        if (disp2)
        {
            if (disp2->fd >= 0)
            {
                if (disp2->mmap_data != MAP_FAILED)
                {
                    munmap((void *)disp2->mmap_data, disp2->len);
                }
                close(disp2->fd);
                free(disp2->data);
            }
            free(disp2);
            disp2 = NULL;
        }
    }

    // Recycle hwframe
    if (dec_hwframe_tracker.ui16FrameIdx)
        ni_hwframe_buffer_recycle(&dec_hwframe_tracker,
                                    dec_hwframe_tracker.device_handle);

    if (scaler_hwframe_tracker.ui16FrameIdx)
        ni_hwframe_buffer_recycle(&scaler_hwframe_tracker,
                                    scaler_hwframe_tracker.device_handle);

    if (pad_hwframe_tracker.ui16FrameIdx)
        ni_hwframe_buffer_recycle(&pad_hwframe_tracker,
                                    pad_hwframe_tracker.device_handle);

    ni_device_session_close(&dec_ctx, 1, NI_DEVICE_TYPE_DECODER);
    ni_device_session_close(&sca_ctx, 1, NI_DEVICE_TYPE_SCALER);
    ni_device_session_close(&pad_ctx, 1, NI_DEVICE_TYPE_SCALER);
    if (fmt_ctx.device_handle != NI_INVALID_DEVICE_HANDLE)
        ni_device_session_close(&fmt_ctx, 1, NI_DEVICE_TYPE_SCALER);

    ni_packet_buffer_free(&(in_pkt.data.packet));
    ni_frame_buffer_free(&(out_frame.data.frame));

end:
    ni_device_session_context_clear(&dec_ctx);
    ni_device_session_context_clear(&sca_ctx);
    ni_device_session_context_clear(&pad_ctx);
    ni_device_session_context_clear(&fmt_ctx);

    free(p_dec_api_param);
    if (p_file)
    {
        fclose(p_file);
    }
    // if(p_scale_out)
    // {
    //     fclose(p_scale_out);
    // }
    if (g_file_cache)
    {
        free(g_file_cache);
        g_file_cache = NULL;
    }

    ni_log(NI_LOG_INFO,"All Done.\n");
    return ret;
}
