// Copyright (c) "2022" Advanced Micro Devices, Inc. All rights reserved.

#pragma once

#include <options.h>

typedef int32_t DmaBufFd;
typedef int32_t SyncFileFd;

class TimestampRecord;

// Type used to index static resources, such as CommandBuffer, RenderTarget image, Swapchain image,
// etc.
struct ResourceIndex {
    ResourceIndex(uint32_t val) : value(val) {}
    ResourceIndex(int val) : value((uint32_t)val) {}
    operator unsigned(void) {
        return value;
    }
    bool operator<(ResourceIndex other) {
        return value < other.value;
    }
    ResourceIndex &operator++(int) {
        value++;
        return *this;
    }

    uint32_t value;
};

// Type used to index dynamic resources, such as in-flight commands, fences, semaphores.
struct FrameIndex {
    FrameIndex(uint32_t val) : value(val) {}
    FrameIndex(int val) : value((uint32_t)val) {}
    operator unsigned(void) {
        return value;
    }
    bool operator<(FrameIndex other) {
        return value < other.value;
    }
    FrameIndex &operator++(int) {
        value++;
        return *this;
    }
    FrameIndex &operator+=(int n) {
        value += n;
        return *this;
    }
    FrameIndex &operator%=(FrameIndex other) {
        value %= other.value;
        return *this;
    }

    uint32_t value;
};

enum class ImagePlaneAspect {
    Plane0,
    Plane1,
    Plane2,
};

class IEncodeDevice {
public:
    IEncodeDevice(){};
    virtual ~IEncodeDevice(){};

    IEncodeDevice(const IEncodeDevice &) = delete;
    IEncodeDevice &operator=(const IEncodeDevice &) = delete;

    virtual void device_init(const Option &) = 0;
    virtual void device_cleanup() = 0;

    virtual FrameIndex get_max_number_of_inflight_commands() = 0;

    virtual void import_semaphore_encoding_wait_semaphore(SyncFileFd syncFd, FrameIndex index) = 0;

    virtual SyncFileFd export_semaphore_encoding_signal_semaphore(FrameIndex index) = 0;

    virtual ResourceIndex get_max_number_of_shared_images() = 0;

    virtual DmaBufFd allocate_shared_image_by_index(ResourceIndex index,
                                                    int32_t width,
                                                    int32_t height) = 0;

    virtual size_t get_shared_image_size_by_index(ResourceIndex index) = 0;

    virtual void get_shared_image_plane_range_by_index(ResourceIndex index,
                                                       ImagePlaneAspect image_aspect,
                                                       size_t *offset,
                                                       size_t *size) = 0;

    virtual ResourceIndex acquire_shared_image(FrameIndex frame_index) = 0;

    virtual void encoding_on_memory(uint32_t cur_frame,
                                    TimestampRecord *p_timestamp_record,
                                    FrameIndex frame_index,
                                    ResourceIndex rsc_index) = 0;
};
