// Modifications Copyright (C)2022 Advanced Micro Devices, Inc. All rights reserved.
// Notified per clause 4(b) of the license

/*
 * Copyright (c) 2015-2019 The Khronos Group Inc.
 * Copyright (c) 2015-2019 Valve Corporation
 * Copyright (c) 2015-2019 LunarG, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Authors: Jeremy Hayes <jeremy@lunarg.com>
 */

#include "vk_common.h"

#include <string_view>

#include <app.h>

vk::Bool32 check_layers(uint32_t check_count,
                        char const *const *const check_names,
                        uint32_t layer_count,
                        vk::LayerProperties *layers) {
    for (uint32_t i = 0; i < check_count; i++) {
        vk::Bool32 found = VK_FALSE;
        for (uint32_t j = 0; j < layer_count; j++) {
            if (!strcmp(check_names[i], layers[j].layerName)) {
                found = VK_TRUE;
                break;
            }
        }
        if (!found) {
            fprintf(stderr, "Cannot find layer: %s\n", check_names[i]);
            return 0;
        }
    }
    return VK_TRUE;
}

#if defined(VK_USE_PLATFORM_DISPLAY_KHR)
int find_display_gpu(int gpu_number,
                     int gpu_no_select,
                     uint32_t gpu_count,
                     std::unique_ptr<vk::PhysicalDevice[]> &physical_devices) {
    uint32_t display_count = 0;
    vk::Result result;
    int gpu_return = gpu_number;
    if (gpu_number >= 0) {
        result = physical_devices[gpu_number].getDisplayPropertiesKHR(&display_count, nullptr);
        VERIFY(result == vk::Result::eSuccess);
    } else {
        for (uint32_t i = 0; i < gpu_count; i++) {
            if (i == gpu_no_select) {
                continue;
            }
            result = physical_devices[i].getDisplayPropertiesKHR(&display_count, nullptr);
            VERIFY(result == vk::Result::eSuccess);
            if (display_count) {
                gpu_return = i;
                break;
            }
        }
    }
    if (display_count > 0)
        return gpu_return;
    else
        return -1;
}
#endif

void init_vulkan_instance_and_phys_device(const bool in_need_validate,
                                          const bool in_need_display,
                                          const int in_force_select_gpu_number,  // -1 to ignore
                                          const int in_force_exclude_gpu_number, // -1 to ignore
                                          int *out_gpu_number,
                                          vk::Instance *out_instance,
                                          uint32_t *out_enabled_device_extension_count,
                                          const char *out_device_extension_names[],
                                          uint32_t *out_enabled_layer_count,
                                          const char *out_layer_names[],
                                          vk::PhysicalDevice *out_gpu,
                                          vk::PhysicalDeviceProperties *out_gpu_props,
                                          uint32_t *out_queue_family_count,
                                          vk::QueueFamilyProperties **out_queue_props) {
    uint32_t instance_extension_count = 0;
    uint32_t instance_layer_count = 0;
    char const *const instance_validation_layers[] = {"VK_LAYER_KHRONOS_validation"};
    uint32_t enabled_extension_count = 0;
    uint32_t enabled_layer_count = 0;
    const char *extension_names[64] = {};
    const char *enabled_layers[64] = {};

    // Look for validation layers
    vk::Bool32 validation_found = VK_FALSE;
    if (in_need_validate) {
        auto result = vk::enumerateInstanceLayerProperties(
            &instance_layer_count, static_cast<vk::LayerProperties *>(nullptr));
        VERIFY(result == vk::Result::eSuccess);

        if (instance_layer_count > 0) {
            std::unique_ptr<vk::LayerProperties[]> instance_layers(
                new vk::LayerProperties[instance_layer_count]);
            result =
                vk::enumerateInstanceLayerProperties(&instance_layer_count, instance_layers.get());
            VERIFY(result == vk::Result::eSuccess);

            validation_found = check_layers(ARRAY_SIZE(instance_validation_layers),
                                            instance_validation_layers,
                                            instance_layer_count,
                                            instance_layers.get());
            if (validation_found) {
                enabled_layer_count = ARRAY_SIZE(instance_validation_layers);
                enabled_layers[0] = "VK_LAYER_KHRONOS_validation";
            }
        }

        if (!validation_found) {
            ERR_EXIT(
                "vkEnumerateInstanceLayerProperties failed to find required validation layer.\n\n"
                "Please look at the Getting Started guide for additional information.\n",
                "vkCreateInstance Failure");
        }
    }

    /* Look for instance extensions */
    vk::Bool32 surfaceExtFound = VK_FALSE;
    vk::Bool32 platformSurfaceExtFound = VK_FALSE;
    memset(extension_names, 0, sizeof(extension_names));

    auto result = vk::enumerateInstanceExtensionProperties(
        nullptr, &instance_extension_count, static_cast<vk::ExtensionProperties *>(nullptr));
    VERIFY(result == vk::Result::eSuccess);

    if (instance_extension_count > 0) {
        std::unique_ptr<vk::ExtensionProperties[]> instance_extensions(
            new vk::ExtensionProperties[instance_extension_count]);
        result = vk::enumerateInstanceExtensionProperties(
            nullptr, &instance_extension_count, instance_extensions.get());
        VERIFY(result == vk::Result::eSuccess);

        for (uint32_t i = 0; i < instance_extension_count; i++) {
            if (!strcmp(VK_KHR_GET_PHYSICAL_DEVICE_PROPERTIES_2_EXTENSION_NAME,
                        instance_extensions[i].extensionName)) {
                extension_names[enabled_extension_count++] =
                    VK_KHR_GET_PHYSICAL_DEVICE_PROPERTIES_2_EXTENSION_NAME;
            }
            if (!strcmp(VK_KHR_SURFACE_EXTENSION_NAME, instance_extensions[i].extensionName)) {
                surfaceExtFound = 1;
                extension_names[enabled_extension_count++] = VK_KHR_SURFACE_EXTENSION_NAME;
            }
#if defined(VK_USE_PLATFORM_WIN32_KHR)
            if (!strcmp(VK_KHR_WIN32_SURFACE_EXTENSION_NAME,
                        instance_extensions[i].extensionName)) {
                platformSurfaceExtFound = 1;
                extension_names[enabled_extension_count++] = VK_KHR_WIN32_SURFACE_EXTENSION_NAME;
            }
#elif defined(VK_USE_PLATFORM_XLIB_KHR)
            if (!strcmp(VK_KHR_XLIB_SURFACE_EXTENSION_NAME, instance_extensions[i].extensionName)) {
                platformSurfaceExtFound = 1;
                extension_names[enabled_extension_count++] = VK_KHR_XLIB_SURFACE_EXTENSION_NAME;
            }
#elif defined(VK_USE_PLATFORM_XCB_KHR)
            if (!strcmp(VK_KHR_XCB_SURFACE_EXTENSION_NAME, instance_extensions[i].extensionName)) {
                platformSurfaceExtFound = 1;
                extension_names[enabled_extension_count++] = VK_KHR_XCB_SURFACE_EXTENSION_NAME;
            }
#elif defined(VK_USE_PLATFORM_WAYLAND_KHR)
            if (!strcmp(VK_KHR_WAYLAND_SURFACE_EXTENSION_NAME,
                        instance_extensions[i].extensionName)) {
                platformSurfaceExtFound = 1;
                extension_names[enabled_extension_count++] = VK_KHR_WAYLAND_SURFACE_EXTENSION_NAME;
            }
#elif defined(VK_USE_PLATFORM_DIRECTFB_EXT)
            if (!strcmp(VK_EXT_DIRECTFB_SURFACE_EXTENSION_NAME,
                        instance_extensions[i].extensionName)) {
                platformSurfaceExtFound = 1;
                extension_names[enabled_extension_count++] = VK_EXT_DIRECTFB_SURFACE_EXTENSION_NAME;
            }
#elif defined(VK_USE_PLATFORM_DISPLAY_KHR)
            if (!strcmp(VK_KHR_DISPLAY_EXTENSION_NAME, instance_extensions[i].extensionName)) {
                platformSurfaceExtFound = 1;
                extension_names[enabled_extension_count++] = VK_KHR_DISPLAY_EXTENSION_NAME;
            }
#elif defined(VK_USE_PLATFORM_METAL_EXT)
            if (!strcmp(VK_EXT_METAL_SURFACE_EXTENSION_NAME,
                        instance_extensions[i].extensionName)) {
                platformSurfaceExtFound = 1;
                extension_names[enabled_extension_count++] = VK_EXT_METAL_SURFACE_EXTENSION_NAME;
            }
#endif
            assert(enabled_extension_count < 64);
        }
    }

    if (!surfaceExtFound) {
        ERR_EXIT("vkEnumerateInstanceExtensionProperties failed to find "
                 "the " VK_KHR_SURFACE_EXTENSION_NAME " extension.\n\n"
                 "Do you have a compatible Vulkan installable client driver (ICD) installed?\n"
                 "Please look at the Getting Started guide for additional information.\n",
                 "vkCreateInstance Failure");
    }

    if (!platformSurfaceExtFound) {
#if defined(VK_USE_PLATFORM_WIN32_KHR)
        ERR_EXIT("vkEnumerateInstanceExtensionProperties failed to find "
                 "the " VK_KHR_WIN32_SURFACE_EXTENSION_NAME " extension.\n\n"
                 "Do you have a compatible Vulkan installable client driver (ICD) installed?\n"
                 "Please look at the Getting Started guide for additional information.\n",
                 "vkCreateInstance Failure");
#elif defined(VK_USE_PLATFORM_XCB_KHR)
        ERR_EXIT("vkEnumerateInstanceExtensionProperties failed to find "
                 "the " VK_KHR_XCB_SURFACE_EXTENSION_NAME " extension.\n\n"
                 "Do you have a compatible Vulkan installable client driver (ICD) installed?\n"
                 "Please look at the Getting Started guide for additional information.\n",
                 "vkCreateInstance Failure");
#elif defined(VK_USE_PLATFORM_WAYLAND_KHR)
        ERR_EXIT("vkEnumerateInstanceExtensionProperties failed to find "
                 "the " VK_KHR_WAYLAND_SURFACE_EXTENSION_NAME " extension.\n\n"
                 "Do you have a compatible Vulkan installable client driver (ICD) installed?\n"
                 "Please look at the Getting Started guide for additional information.\n",
                 "vkCreateInstance Failure");
#elif defined(VK_USE_PLATFORM_XLIB_KHR)
        ERR_EXIT("vkEnumerateInstanceExtensionProperties failed to find "
                 "the " VK_KHR_XLIB_SURFACE_EXTENSION_NAME " extension.\n\n"
                 "Do you have a compatible Vulkan installable client driver (ICD) installed?\n"
                 "Please look at the Getting Started guide for additional information.\n",
                 "vkCreateInstance Failure");
#elif defined(VK_USE_PLATFORM_DIRECTFB_EXT)
        ERR_EXIT("vkEnumerateInstanceExtensionProperties failed to find "
                 "the " VK_EXT_DIRECTFB_SURFACE_EXTENSION_NAME " extension.\n\n"
                 "Do you have a compatible Vulkan installable client driver (ICD) installed?\n"
                 "Please look at the Getting Started guide for additional information.\n",
                 "vkCreateInstance Failure");
#elif defined(VK_USE_PLATFORM_DISPLAY_KHR)
        ERR_EXIT("vkEnumerateInstanceExtensionProperties failed to find "
                 "the " VK_KHR_DISPLAY_EXTENSION_NAME " extension.\n\n"
                 "Do you have a compatible Vulkan installable client driver (ICD) installed?\n"
                 "Please look at the Getting Started guide for additional information.\n",
                 "vkCreateInstance Failure");
#elif defined(VK_USE_PLATFORM_METAL_EXT)
        ERR_EXIT("vkEnumerateInstanceExtensionProperties failed to find "
                 "the " VK_EXT_METAL_SURFACE_EXTENSION_NAME
                 " extension.\n\nDo you have a compatible "
                 "Vulkan installable client driver (ICD) installed?\nPlease "
                 "look at the Getting Started guide for additional "
                 "information.\n",
                 "vkCreateInstance Failure");
#endif
    }
    auto const app = vk::ApplicationInfo()
                         .setPApplicationName(APP_SHORT_NAME)
                         .setApplicationVersion(0)
                         .setPEngineName(APP_SHORT_NAME)
                         .setEngineVersion(0)
                         .setApiVersion(VK_API_VERSION_1_2);
    auto const inst_info = vk::InstanceCreateInfo()
                               .setPApplicationInfo(&app)
                               .setEnabledLayerCount(enabled_layer_count)
                               .setPpEnabledLayerNames(instance_validation_layers)
                               .setEnabledExtensionCount(enabled_extension_count)
                               .setPpEnabledExtensionNames(extension_names);

    vk::Instance inst = {};
    result = vk::createInstance(&inst_info, nullptr, &inst);
    if (result == vk::Result::eErrorIncompatibleDriver) {
        ERR_EXIT("Cannot find a compatible Vulkan installable client driver (ICD).\n\n"
                 "Please look at the Getting Started guide for additional information.\n",
                 "vkCreateInstance Failure");
    } else if (result == vk::Result::eErrorExtensionNotPresent) {
        ERR_EXIT("Cannot find a specified extension library.\n"
                 "Make sure your layers path is set appropriately.\n",
                 "vkCreateInstance Failure");
    } else if (result != vk::Result::eSuccess) {
        ERR_EXIT("vkCreateInstance failed.\n\n"
                 "Do you have a compatible Vulkan installable client driver (ICD) installed?\n"
                 "Please look at the Getting Started guide for additional information.\n",
                 "vkCreateInstance Failure");
    }

    /* Make initial call to query gpu_count, then second call for gpu info */
    uint32_t gpu_count = 0;
    result = inst.enumeratePhysicalDevices(&gpu_count, static_cast<vk::PhysicalDevice *>(nullptr));
    VERIFY(result == vk::Result::eSuccess);

    if (gpu_count <= 0) {
        ERR_EXIT("vkEnumeratePhysicalDevices reported zero accessible devices.\n\n"
                 "Do you have a compatible Vulkan installable client driver (ICD) installed?\n"
                 "Please look at the Getting Started guide for additional information.\n",
                 "vkEnumeratePhysicalDevices Failure");
    }

    std::unique_ptr<vk::PhysicalDevice[]> physical_devices(new vk::PhysicalDevice[gpu_count]);
    result = inst.enumeratePhysicalDevices(&gpu_count, physical_devices.get());
    VERIFY(result == vk::Result::eSuccess);

    if (in_force_select_gpu_number >= 0 && !((uint32_t)in_force_select_gpu_number < gpu_count)) {
        fprintf(stderr,
                "GPU %d specified is not present, GPU count = %u\n",
                in_force_select_gpu_number,
                gpu_count);
        ERR_EXIT("Specified GPU number is not present", "User Error");
    }

    int gpu_number = in_force_select_gpu_number;
#if defined(VK_USE_PLATFORM_DISPLAY_KHR)
    if (in_need_display) {
        gpu_number =
            find_display_gpu(gpu_number, in_force_exclude_gpu_number, gpu_count, physical_devices);
        if (gpu_number < 0) {
            printf("Cannot find any display!\n");
            fflush(stdout);
            exit(1);
        }
    }
#else
    /* Try to auto select most suitable device */
    if (gpu_number == -1) {
        uint32_t count_device_type[VK_PHYSICAL_DEVICE_TYPE_CPU + 1];
        memset(count_device_type, 0, sizeof(count_device_type));

        for (uint32_t i = 0; i < gpu_count; i++) {
            if (i != in_force_exclude_gpu_number) {
                const auto physicalDeviceProperties = physical_devices[i].getProperties();
                assert(static_cast<int>(physicalDeviceProperties.deviceType) <=
                       VK_PHYSICAL_DEVICE_TYPE_CPU);
                count_device_type[static_cast<int>(physicalDeviceProperties.deviceType)]++;
            }
        }

        const vk::PhysicalDeviceType device_type_preference[] = {
            vk::PhysicalDeviceType::eDiscreteGpu,
            vk::PhysicalDeviceType::eIntegratedGpu,
            vk::PhysicalDeviceType::eVirtualGpu,
            vk::PhysicalDeviceType::eCpu,
            vk::PhysicalDeviceType::eOther};
        vk::PhysicalDeviceType search_for_device_type = vk::PhysicalDeviceType::eDiscreteGpu;
        for (uint32_t i = 0; i < sizeof(device_type_preference) / sizeof(vk::PhysicalDeviceType);
             i++) {
            if (count_device_type[static_cast<int>(device_type_preference[i])]) {
                search_for_device_type = device_type_preference[i];
                break;
            }
        }

        for (uint32_t i = 0; i < gpu_count; i++) {
            const auto physicalDeviceProperties = physical_devices[i].getProperties();
            if (i != in_force_exclude_gpu_number &&
                physicalDeviceProperties.deviceType == search_for_device_type) {
                gpu_number = i;
                break;
            }
        }
    }
#endif
    assert(gpu_number >= 0);
    vk::PhysicalDevice gpu = physical_devices[gpu_number];
    {
        auto physicalDeviceProperties = gpu.getProperties();
        fprintf(stderr,
                "Selected GPU %d: %s, type: %s\n",
                gpu_number,
                physicalDeviceProperties.deviceName.data(),
                to_string(physicalDeviceProperties.deviceType).c_str());
    }
    physical_devices.reset();

    /* Look for device extensions */
    uint32_t device_extension_count = 0;
    vk::Bool32 swapchainExtFound = VK_FALSE;
    vk::Bool32 externalSemaphoreKhrFound = VK_FALSE;
    vk::Bool32 externalMemoryDmaBufExtFound = VK_FALSE;
    vk::Bool32 dedicateAllocExtFound = VK_FALSE;
    vk::Bool32 SamplerYCbCrKhrFound = VK_FALSE;
    enabled_extension_count = 0;
    memset(extension_names, 0, sizeof(extension_names));

    result = gpu.enumerateDeviceExtensionProperties(
        nullptr, &device_extension_count, static_cast<vk::ExtensionProperties *>(nullptr));
    VERIFY(result == vk::Result::eSuccess);

    if (device_extension_count > 0) {
        std::unique_ptr<vk::ExtensionProperties[]> device_extensions(
            new vk::ExtensionProperties[device_extension_count]);
        result = gpu.enumerateDeviceExtensionProperties(
            nullptr, &device_extension_count, device_extensions.get());
        VERIFY(result == vk::Result::eSuccess);

        for (uint32_t i = 0; i < device_extension_count; i++) {
            if (!strcmp(VK_KHR_SWAPCHAIN_EXTENSION_NAME, device_extensions[i].extensionName)) {
                swapchainExtFound = 1;
                extension_names[enabled_extension_count++] = VK_KHR_SWAPCHAIN_EXTENSION_NAME;
            }
            if (!strcmp("VK_KHR_portability_subset", device_extensions[i].extensionName)) {
                extension_names[enabled_extension_count++] = "VK_KHR_portability_subset";
            }
            if (!strcmp(VK_KHR_EXTERNAL_MEMORY_CAPABILITIES_EXTENSION_NAME,
                        device_extensions[i].extensionName)) {
                extension_names[enabled_extension_count++] =
                    VK_KHR_EXTERNAL_MEMORY_CAPABILITIES_EXTENSION_NAME;
            }
            if (!strcmp(VK_KHR_EXTERNAL_MEMORY_EXTENSION_NAME,
                        device_extensions[i].extensionName)) {
                extension_names[enabled_extension_count++] = VK_KHR_EXTERNAL_MEMORY_EXTENSION_NAME;
            }
            if (!strcmp(VK_KHR_EXTERNAL_MEMORY_FD_EXTENSION_NAME,
                        device_extensions[i].extensionName)) {
                extension_names[enabled_extension_count++] =
                    VK_KHR_EXTERNAL_MEMORY_FD_EXTENSION_NAME;
            }
            if (!strcmp(VK_KHR_GET_MEMORY_REQUIREMENTS_2_EXTENSION_NAME,
                        device_extensions[i].extensionName)) {
                extension_names[enabled_extension_count++] =
                    VK_KHR_GET_MEMORY_REQUIREMENTS_2_EXTENSION_NAME;
            }
            if (!strcmp(VK_KHR_DEDICATED_ALLOCATION_EXTENSION_NAME,
                        device_extensions[i].extensionName)) {
                dedicateAllocExtFound = 1;
                extension_names[enabled_extension_count++] =
                    VK_KHR_DEDICATED_ALLOCATION_EXTENSION_NAME;
            }
            if (!strcmp(VK_EXT_EXTERNAL_MEMORY_DMA_BUF_EXTENSION_NAME,
                        device_extensions[i].extensionName)) {
                externalMemoryDmaBufExtFound = 1;
                extension_names[enabled_extension_count++] =
                    VK_EXT_EXTERNAL_MEMORY_DMA_BUF_EXTENSION_NAME;
            }
            if (!strcmp(VK_KHR_EXTERNAL_SEMAPHORE_CAPABILITIES_EXTENSION_NAME,
                        device_extensions[i].extensionName)) {
                extension_names[enabled_extension_count++] =
                    VK_KHR_EXTERNAL_SEMAPHORE_CAPABILITIES_EXTENSION_NAME;
            }
            if (!strcmp(VK_KHR_EXTERNAL_SEMAPHORE_EXTENSION_NAME,
                        device_extensions[i].extensionName)) {
                extension_names[enabled_extension_count++] =
                    VK_KHR_EXTERNAL_SEMAPHORE_EXTENSION_NAME;
            }
            if (!strcmp(VK_KHR_EXTERNAL_SEMAPHORE_FD_EXTENSION_NAME,
                        device_extensions[i].extensionName)) {
                externalSemaphoreKhrFound = 1;
                extension_names[enabled_extension_count++] =
                    VK_KHR_EXTERNAL_SEMAPHORE_FD_EXTENSION_NAME;
            }
            if (!strcmp(VK_KHR_SAMPLER_YCBCR_CONVERSION_EXTENSION_NAME,
                        device_extensions[i].extensionName)) {
                SamplerYCbCrKhrFound = 1;
                extension_names[enabled_extension_count++] =
                    VK_KHR_SAMPLER_YCBCR_CONVERSION_EXTENSION_NAME;
            }
            if (!strcmp(VK_KHR_MAINTENANCE_1_EXTENSION_NAME, device_extensions[i].extensionName)) {
                extension_names[enabled_extension_count++] = VK_KHR_MAINTENANCE_1_EXTENSION_NAME;
            }
            if (!strcmp(VK_KHR_BIND_MEMORY_2_EXTENSION_NAME, device_extensions[i].extensionName)) {
                extension_names[enabled_extension_count++] = VK_KHR_BIND_MEMORY_2_EXTENSION_NAME;
            }
            assert(enabled_extension_count < 64);
        }
    }

    if (in_need_display && !swapchainExtFound) {
        ERR_EXIT("vkEnumerateDeviceExtensionProperties failed to find "
                 "the " VK_KHR_SWAPCHAIN_EXTENSION_NAME " extension.\n\n"
                 "Do you have a compatible Vulkan installable client driver (ICD) installed?\n"
                 "Please look at the Getting Started guide for additional information.\n",
                 "vkCreateInstance Failure");
    }

    if (!externalMemoryDmaBufExtFound) {
        ERR_EXIT("vkEnumerateDeviceExtensionProperties failed to find "
                 "the " VK_EXT_EXTERNAL_MEMORY_DMA_BUF_EXTENSION_NAME " extension.\n\n"
                 "The extension is required on the source gpu for P2P DMA memory write.",
                 "Required Extension missing");
    }
    if (!dedicateAllocExtFound) {
        ERR_EXIT("vkEnumerateDeviceExtensionProperties failed to find "
                 "the " VK_KHR_DEDICATED_ALLOCATION_EXTENSION_NAME " extension.\n\n"
                 "The extension is required on the allocating image memory for P2P DMA.",
                 "Required Extension missing");
    }
    if (!externalSemaphoreKhrFound) {
        ERR_EXIT("vkEnumerateDeviceExtensionProperties failed to find "
                 "the " VK_KHR_EXTERNAL_SEMAPHORE_FD_EXTENSION_NAME " extension.\n\n"
                 "The extension is required on the synchronize P2P DMA.",
                 "Required Extension missing");
    }
    if (!SamplerYCbCrKhrFound) {
        ERR_EXIT("vkEnumerateDeviceExtensionProperties failed to find "
                 "the " VK_KHR_SAMPLER_YCBCR_CONVERSION_EXTENSION_NAME " extension.\n\n"
                 "The extension is required to verify YCbCr image copy.",
                 "Required Extension missing");
    }
    gpu.getProperties(out_gpu_props);

    /* Call with nullptr data to get count */
    uint32_t queue_family_count = 0;
    gpu.getQueueFamilyProperties(&queue_family_count,
                                 static_cast<vk::QueueFamilyProperties *>(nullptr));
    assert(queue_family_count >= 1);

    *out_queue_props = new vk::QueueFamilyProperties[queue_family_count];
    gpu.getQueueFamilyProperties(&queue_family_count, *out_queue_props);
    *out_queue_family_count = queue_family_count;

    // Query fine-grained feature support for this device.
    //  If app has specific feature requirements it should check supported
    //  features based on this query
    vk::PhysicalDeviceFeatures physDevFeatures;
    gpu.getFeatures(&physDevFeatures);

    *out_enabled_device_extension_count = enabled_extension_count;
    memcpy(out_device_extension_names,
           extension_names,
           sizeof(const char *) * enabled_extension_count);
    *out_enabled_layer_count = enabled_layer_count;
    memcpy(out_layer_names, enabled_layers, sizeof(const char *) * enabled_layer_count);

    *out_gpu_number = gpu_number;
    *out_gpu = gpu;
    *out_instance = inst;
}

bool memory_type_from_properties(const vk::PhysicalDeviceMemoryProperties &memory_properties,
                                 uint32_t typeBits,
                                 vk::MemoryPropertyFlags requirements_mask,
                                 uint32_t *typeIndex) {
    // Search memtypes to find first index with those properties
    for (uint32_t i = 0; i < VK_MAX_MEMORY_TYPES; i++) {
        if ((typeBits & 1) == 1) {
            // Type is available, does it match user properties?
            if ((memory_properties.memoryTypes[i].propertyFlags & requirements_mask) ==
                requirements_mask) {
                *typeIndex = i;
                return true;
            }
        }
        typeBits >>= 1;
    }

    // No memory types matched, return failure
    return false;
}
