// Copyright (c) 2022-2023 Advanced Micro Devices, Inc. All rights reserved.

#include <signal.h>
#include <sys/time.h>
#include <unistd.h>
#include <chrono>
#include <thread>

#include "cube.h"
#include "gettime.h"
#include "netint_encoding.h"
#include "options.h"
#include "timestamp_record.h"

static bool is_offscreen_running = true;

void sig_handler(int signo) {
    if (signo == SIGINT) {
        is_offscreen_running = false;
    }
}

void offscreen_draw(TimestampRecord *p_timestamp_record,
                    uint32_t frame_count,
                    const std::unique_ptr<ISurfaceRenderContext> &renderer) {
    // If you are using offscreen with NetInt, change FPS needs to change NetInt
    // config as well.
    constexpr int frame_per_seconds = 30;

    if (signal(SIGINT, sig_handler) == SIG_ERR) {
        fprintf(stderr, "Not able to register SIGINT handler!\n");
        return;
    }

    std::chrono::steady_clock::duration second = std::chrono::seconds{1};
    std::chrono::steady_clock::duration frame_time = second / frame_per_seconds;
    std::chrono::steady_clock::time_point target;
    for (uint32_t i = 0; is_offscreen_running && i < frame_count; ++i) {
        target = std::chrono::steady_clock::now() + frame_time;
        renderer->draw(i, p_timestamp_record);
        std::this_thread::sleep_until(target);
    }
}

int main(int argc, char **argv) {
    Option option = parse_arguments_to_options(argc, argv);

    std::unique_ptr<ISurfaceRenderContext> rendering_context = {};
    std::unique_ptr<IEncodeDevice> encoding_device = {};

    NetintEncoding *netint_encoding = new NetintEncoding();
    encoding_device.reset(static_cast<IEncodeDevice *>(netint_encoding));
    encoding_device->device_init(option);

    Cube *cube = new Cube(option);
    cube->set_encoding_device(encoding_device.get());
    rendering_context.reset(static_cast<ISurfaceRenderContext *>(cube));
    rendering_context->init();

    TimestampRecord timestamp_record(option);

    offscreen_draw(&timestamp_record, option.frameCount, rendering_context);

    if (encoding_device.get() != nullptr) {
        encoding_device->device_cleanup();
    }
    rendering_context->cleanup();

    timestamp_record.report();

    return 0;
}
