#include <linux/dma-buf.h>
#include <unistd.h>

#include <iostream>
#include <string>
#include <vector>
#include <unordered_set>

extern "C"
{
#include <ni_device_api.h>
}

#include "p2p_gl_gl.h"
#include "p2p_gl_egl.h"
#include "p2p_gl_xcoder.h"

static int draw(std::shared_ptr<GL_rgba_base> draw_to, int w, int h)
{
    static int count = 0;
    static std::vector<double> color{1.0, 1.0, 0.0, 1.0}; // RGBA

    ++count;

    if (count % 30 == 0)
    {
        int back = color[color.size() - 2];
        for (int i = color.size() - 2; i > 0; --i)
        {
            color[i] = color[i - 1];
        }
        color[0] = back;
    }

    if(!draw_to->make_current())
    {
        std::cerr<<"Failed to make_current draw to\n";
        return -1;
    }
    glViewport(0, 0, w, h);
    glClearColor(color[0], color[1], color[2], color[3]);
    glClear(GL_COLOR_BUFFER_BIT);
    glFinish(); // or glFlush();
    if (glCheckFramebufferStatus(GL_FRAMEBUFFER) != GL_FRAMEBUFFER_COMPLETE)
    {
        std::cerr << "Not finish\n";
        return -1;
    }
    return 0;
}

int main(int argc, const char *argv[])
{

    const int w = 1920;
    const int h = 1080;
    const int POOL_SIZE = 2;
    const int total_frames = 300;
    int xcoder_uid = 0;
    const ni_codec_format_t out_format = NI_CODEC_FORMAT_H265;

    ni_pix_fmt_t pixel_format = NI_PIX_FMT_RGBA;

    int gl_util = 0; // 0 glew, 1 egl

    if (argc >= 2)
    {
        if(argc > 2)
        {
            std::cerr << "Too many patameters. Will ignore the parameters after the first one.\n";
        }
        if (strcmp(argv[1], "egl") == 0)
        {
            gl_util = 1;
        }
        else if (strcmp(argv[1], "glew") == 0)
        {
            gl_util = 0;
        }
        else
        {
            std::cerr << "Invalid parameter: " << argv[1] << ". Using default value glew instead" << "\n";
            gl_util = 0;
        }
    }

    switch (gl_util)
    {
    case 0:
    {
        pixel_format = NI_PIX_FMT_RGBA;
        GLFW_GLEW_EXT_INIT &init = GLFW_GLEW_EXT_INIT::init_gwfw_glew_ext();
        if (!init.initialized())
        {
            std::cerr << "Failed to initialize glfw or glew\n";
            return -1;
        }
        break;
    }

    case 1:
    {
        pixel_format = NI_PIX_FMT_BGRA;
        break;
    }     
    }

    ni_session_context_t enc_ctx{};
    ni_session_context_t upl_ctx{};
    ni_xcoder_params_t enc_api_param;
    ni_session_data_io_t out_packet{};
    ni_frame_t p2p_frame[POOL_SIZE];
    int ret = -1;

    if (ni_device_session_context_init(&enc_ctx) < 0)
    {
        std::cerr << "Error: init encoder context error\n";
        return -1;
    }

    if (ni_device_session_context_init(&upl_ctx) < 0)
    {
        std::cerr << "Error: init uploader context error\n";
        return -1;
    }

    uploader_open_session_gl_test(&upl_ctx, &xcoder_uid, w, h, pixel_format, POOL_SIZE);

    if (ni_encoder_init_default_params(&enc_api_param, 30, 1, 200000, w,
                                       h, static_cast<ni_codec_format_t>(enc_ctx.codec_format)) < 0)
    {
        fprintf(stderr, "Error: encoder init default set up error\n");
        ni_device_session_close(&upl_ctx, 1, NI_DEVICE_TYPE_UPLOAD);
        return -1;
    }

    // For P2P demo, change some of the encoding parameters from
    // the default. Enable low delay encoding.
    if ((ret = ni_encoder_params_set_value(&enc_api_param, "lowDelay", "1")) !=
        NI_RETCODE_SUCCESS)
    {
        fprintf(stderr, "Error: can't set low delay mode %d\n", ret);
        ni_device_session_close(&upl_ctx, 1, NI_DEVICE_TYPE_UPLOAD);
        return -1;
    }

    // Use a GOP preset of 9 which represents a GOP pattern of
    // IPPPPPPP....This will be low latency.
    if ((ret = ni_encoder_params_set_value(&enc_api_param, "gopPresetIdx", "9")) !=
        NI_RETCODE_SUCCESS)
    {
        fprintf(stderr, "Error: can't set gop preset %d\n", ret);
        ni_device_session_close(&upl_ctx, 1, NI_DEVICE_TYPE_UPLOAD);
        return -1;
    }

    // Prepare two frames for double buffering
    ret = p2p_prepare_frames_gl_test(&upl_ctx, w, h,
                                     p2p_frame, POOL_SIZE);

    if (ret < 0)
    {
        ni_device_session_close(&upl_ctx, 1, NI_DEVICE_TYPE_UPLOAD);
        std::cerr << "failed to get p2p frames\n";
        return -1;
    }

    // Open the encoder session with given parameters
    ret = encoder_open_session_gl_test(&enc_ctx, pixel_format, out_format, xcoder_uid,
                                       &enc_api_param, w, h,
                                       &p2p_frame[0]);

    if (ret < 0)
    {
        recycle_frames_gl_test(p2p_frame, POOL_SIZE);
        ni_device_session_close(&upl_ctx, 1, NI_DEVICE_TYPE_UPLOAD);
        std::cerr << "failed to open encoder session\n";
        return -1;
    }

    // int fd = get_dma_buf2();

    std::vector<std::shared_ptr<GL_rgba_base>> rgba_test;

    int render_index = 0;
    int encode_index = -1;

    int send_fin_flag = 0;
    int receive_fin_flag = 0;
    int eos_sent = 0;
    int input_exhausted = 0;
    int need_to_resend = 0;
    int frame_count = 0;

    FILE *p_file = NULL;

    switch (gl_util)
    {
    case 0:
    {
        std::shared_ptr<GLFW_window> window = std::make_shared<GLFW_window>(w, h);
        if (!window->initialized())
        {
            std::cerr << "Failed to initialize window\n";
            ret = -1;
            goto end;
        }
        for (int i = 0; i < POOL_SIZE; ++i)
        {
            int fd = get_fd_from_ni_frame(&p2p_frame[i], true);// glImportMemoryFdEXT() will close the fd inside
            if (fd < 0)
            {
                ret = -1;
                std::cerr << "Failed to get fd\n";
                goto end;
            }
            rgba_test.push_back(std::make_shared<GL_rgba_glew>(fd, w, h, window));
            if (!rgba_test.back()->initialized())
            {
                ret = -1;
                std::cerr << "Failed to get GL_rgba_glew instance\n";
                goto end;
            }
        }
        break;
    }

    case 1:
    {
        std::shared_ptr<GL_egl_display> display = std::make_shared<GL_egl_display>();
        if (!display->initialized())
        {
            std::cerr << "Failed to get display\n";
            ret = -1;
            goto end;
        }
        for (int i = 0; i < POOL_SIZE; ++i)
        {
            int fd = get_fd_from_ni_frame(&p2p_frame[i], false);
            if (fd < 0)
            {
                ret = -1;
                std::cerr << "Failed to get fd\n";
                goto end;
            }
            rgba_test.push_back(std::make_shared<GL_rgba_egl>(fd, w, h, display));
            if (!rgba_test.back()->initialized())
            {
                ret = -1;
                std::cerr << "Failed to get GL_rgba_egl instance\n";
                goto end;
            }
        }
        break;
    }
    }

    std::cout << "Successfully created GL_rgba object" << std::endl;

    p_file = fopen("p2p_gl_test_out.h265", "wb");
    if (!p_file)
    {
        ret = -1;
        goto end;
    }

    std::cout << "Successfully opened output file" << std::endl;

    std::cout << "Encoding start" << std::endl;

    while (send_fin_flag == 0 || receive_fin_flag == 0)
    {
        encode_index = render_index;

        // Fill the frame buffer while the previous frame is being encoded
        if (!input_exhausted && need_to_resend == 0)
        {
            ret = draw(rgba_test[render_index], w, h);
            if (ret < 0)
            {
                goto end;
            }
            ++frame_count;
            if (frame_count == total_frames)
            {
                input_exhausted = 1;
            }
        }

#ifndef _WIN32
        // Lock the hardware frame prior to encoding. This call will
        // block until the frame is ready then lock the frame for use.
        ni_uploader_frame_buffer_lock(&upl_ctx, &p2p_frame[encode_index]);
#endif

        // Encode the frame
        send_fin_flag = encoder_encode_frame_gl_test(&enc_ctx, &p2p_frame[encode_index],
                                                     input_exhausted, need_to_resend, eos_sent);

        // Error, exit
        if (send_fin_flag == 2)
        {
            break;
        }

        // Receive encoded packet data from the encoder
        receive_fin_flag = encoder_receive_data_gl_test(
            &enc_ctx, &out_packet, p_file);

#ifndef _WIN32
        // Unlock the encoded frame now that the compressed packet data has
        // been received. This frame can be re-used.
        ret = ni_uploader_frame_buffer_unlock(&upl_ctx, &p2p_frame[encode_index]);
        if (ret < 0)
        {
            fprintf(stderr, "Failed to unlock frame\n");
            goto end;
        }
#endif

        // Error or eos
        if (receive_fin_flag < 0 || out_packet.data.packet.end_of_stream)
        {
            break;
        }

        // Switch to the other hw frame buffer
        ++render_index;
        if (render_index == POOL_SIZE)
        {
            render_index = 0;
        }
    }

    ret = 0;
end:
    recycle_frames_gl_test(p2p_frame, POOL_SIZE);

    for (int i = 0; i < POOL_SIZE; i++)
    {
        ni_frame_buffer_free(&(p2p_frame[i]));
    }

    ni_packet_buffer_free(&(out_packet.data.packet));

    ni_device_session_close(&upl_ctx, 1, NI_DEVICE_TYPE_UPLOAD);
    ni_device_session_close(&enc_ctx, 1, NI_DEVICE_TYPE_ENCODER);

    //here all fds are already initialized before
    std::unordered_set<int> fd_set;
    fd_set.insert(upl_ctx.blk_io_handle);
    fd_set.insert(upl_ctx.device_handle);
    fd_set.insert(enc_ctx.blk_io_handle);
    fd_set.insert(enc_ctx.device_handle);
    for (int fd : fd_set)
    {
        if (fd >= 0)
        {
            ni_device_close(fd);
        }
    }

    ni_device_session_context_clear(&enc_ctx);
    ni_device_session_context_clear(&upl_ctx);

    if (p_file)
    {
        fclose(p_file);
    }

    return ret;
}